// 私だけの電子オルゴール MyMelo2
// https://koyama.verse.jp/elecraft/mymelo
// 20050307-20240325 koyama88@cameo.plala.or.jp
// (特徴)
// (1) 楽譜を入力してオリジナルの電子オルゴールを作ることができます。
// (2) 単音だけでなく、2和音や3和音など伴奏付の電子オルゴールを作れます。
// (3) 単調なブザー音ではなく、ピアノの音に似たきれいな音色です(音色は3種類)。
// (4) 複数の曲を登録し、ボタン操作で再生する曲を選択できます。
// (5) ArduinoUNO/NANO, Digispark, ATtiny85, ATtiny2313(A)/4313, ATmega328P などで動作します。
//
// (楽譜データの作成)
// (1) 「テキスト音楽サクラ」(またはメモ帳などのテキストエディタ)で「ドドソソララソー」のように楽譜ファイル(〇〇.mml)を作成します(日本語コードはシフトJIS)。
//		使える音はRA1_～SI8およびREST(休符)の87個から最大31種類です。
//		使える音符はL1_(付点全音符)～L32(32分音符)の11個から最大8種類です。
// (2) はじめに mymelo2.wsf をダブルクリックして楽譜データを初期化します。その際、ダイアログボックスが現れるので、矩形波、正弦波、正弦波(伴奏は三角波)のいずれかを選択します。楽譜データファイル(mymelo2フォルダのmymelo2.h)が初期化されます。
// (3) 楽譜ファイル(〇〇.mml)を mymelo2.wsf にドラッグ＆ドロップすると、楽譜データファイルに楽譜データが記録されます。
// (4) (3)の操作を繰り返すと、楽譜データファイルに複数の楽譜を登録できます。
// (5) 楽譜を変更した場合は、(2)の初期化からやり直します。
//
// (プログラムの書き込み)
// ArduinoIDEを起動し、mymelo2フォルダの mymelo2.ino を開いて書き込みます。
//
// (使い方)
// 電源を投入すると、オルゴールの音が再生されます。外部スイッチをつなぐと、スイッチを押したときに再生されます。スイッチの連打または長押しで2曲目以降が再生されます。

#include <avr/sleep.h>
#include "mymelo2.h"

#if F_CPU>=16000000
#define	Ts	32			// サンプリング周期 Ts=32us
#define CPI	71			// 1000000/440/Ts ピッ(440Hz)のカウント
#else
#define	Ts	64			// クロックが遅いと処理が間に合わないのでTs=64usにした
#define CPI	35			// 1000000/440/Ts ピッ(440Hz)のカウント
#endif

#define Th	8000		// チャタリング防止のためのbtnA, btnBのホールドタイム 8ms
#define C5	(500000/Ts)	// 0.5sカウント
#define Ca	(4096/Ts)		// アタックタイム(4ms)カウント

#if defined(__AVR_ATmega168__)||defined(__AVR_ATmega328P__)||defined(__AVR_ATmega328__)
#define __MegaX8__
#define GIMSK	EIMSK
#define TIMSK	TIMSK0
#define btnApin	2		// INT0/PCINT18
#define btnBpin	3		// INT1/PCINT19
#define Bzz1pin	9		// OC1A
#define Bzz2pin	10	// OC1B
#elif defined(ARDUINO_AVR_ATTINYX313)
#define __TinyX313__
#define btnApin	4		// INT0/PCINT13
#define btnBpin	5		// INT1/PCINT14
#define Bzz1pin	12	// OC1A
#define Bzz2pin	13	// OC1B
#elif defined(__AVR_ATtiny25__)||defined(__AVR_ATtiny45__)||defined(__AVR_ATtiny85__)
#define __TinyX5__
#define btnApin 2		// INT0/PCINT2
#define btnBpin 0		// PCINT0
#define Bzz1pin	1		// OC1A
#define Bzz2pin	4		// OC1B
#endif								// -------------------------------------------------------
#define _len(i,n)	pgm_read_byte(&((uint8_t*)pgm_read_ptr(&pLen[i]))[n])
#define _Part1(i,p)	pgm_read_byte(&((uint8_t*)pgm_read_ptr(&pPart1[i]))[p])
#define _Part2(i,p)	pgm_read_byte(&((uint8_t*)pgm_read_ptr(&pPart2[i]))[p])
#define _Part3(i,p)	pgm_read_byte(&((uint8_t*)pgm_read_ptr(&pPart3[i]))[p])
volatile uint8_t btnAcntr=0, btnBcntr=0;	// チャタリング防止カウンタ 32*250=8ms
volatile uint16_t c5=0;					// 0.5秒カウンタ(volatile必要!!)
uint8_t S;											// ボタンの状態
uint8_t imelo=0;								// 曲番号(初期値は0)
uint8_t cdecay;									// 8ms毎に包絡を減衰
uint16_t CNT, cntr;							// Ts*(1953*120/tempo)=62.5ms
uint16_t p1, p2, p3;						// 楽譜part1～part3のポインタ
uint8_t note1,note2,note3;			// 楽譜データDO|L4
uint8_t len1, len2, len3;				// 音符の長さ
uint16_t v,v1,v2,v3, Va,Vb;			// 振幅とその合算値
uint16_t PIcntr=0;							// 20240314 ピッの長さ
uint8_t cPI=0;									// 20240314 ピッのカウント

#if WAVEFORM==0				// --- 減衰矩形波 ----------------------------------------
#define _Cnt(i,n)	pgm_read_word(&((uint16_t*)pgm_read_ptr(&pCnt[i]))[n])
#define	V1	0xffff
#define	V2	0x7fff
#define	V3	0x7fff
uint16_t c1,c2,c3, C1,C2,C3;
#else									// --- 減衰正弦波または減衰正弦波(伴奏は三角波) ----------
// *** SWの表は Ts=32us のものなので「*(Ts/32)」***
#define _SW(i,n)	pgm_read_word(&((uint16_t*)pgm_read_ptr(&pSW[i]))[n])*(Ts/32)
uint8_t len1a=0,len2a=0,len3a=0;	// アタックタイム 4ms
uint16_t E,E1,E2,E3;							// 包絡
uint16_t pW1,pW2,pW3, sw1,sw2,sw3;// 波形ポインタとサンプリングステップ幅
// 正弦波の波形テーブル(値:0～255, データ数:256)
const uint8_t Ws[] PROGMEM={0,0,0,0,1,1,1,2,2,3,4,5,5,6,7,9,10,11,12,14,15,17,18,20,21,23,25,27,29,31,33,35,37,40,42,44,47,49,52,54,57,59,62,65,67,70,73,76,79,82,85,88,90,93,97,100,103,106,109,112,115,118,121,124,127,131,134,137,140,143,146,149,152,155,158,162,165,167,170,173,176,179,182,185,188,190,193,196,198,201,203,206,208,211,213,215,218,220,222,224,226,228,230,232,234,235,237,238,240,241,243,244,245,246,248,249,250,250,251,252,253,253,254,254,254,255,255,255,255,255,255,255,254,254,254,253,253,252,251,250,250,249,248,246,245,244,243,241,240,238,237,235,234,232,230,228,226,224,222,220,218,215,213,211,208,206,203,201,198,196,193,190,188,185,182,179,176,173,170,167,165,162,158,155,152,149,146,143,140,137,134,131,128,124,121,118,115,112,109,106,103,100,97,93,90,88,85,82,79,76,73,70,67,65,62,59,57,54,52,49,47,44,42,40,37,35,33,31,29,27,25,23,21,20,18,17,15,14,12,11,10,9,7,6,5,5,4,3,2,2,1,1,1,0,0,0};
#define _Ws(x)	(uint16_t)pgm_read_byte(&Ws[(uint8_t)((x)>>8)])
#if WAVEFORM==2				// --- 減衰正弦波(伴奏は三角波) -------------------------
// 三角波の波形テーブル(値:0～255, データ数:256)...伴奏用
const uint8_t Wt[] PROGMEM={0,2,4,6,8,10,12,14,16,18,20,22,24,26,28,30,32,34,36,38,40,42,44,46,48,50,52,54,56,58,60,62,64,66,68,70,72,74,76,78,80,82,84,86,88,90,92,94,96,98,100,102,104,106,108,110,112,114,116,118,120,122,124,126,128,129,131,133,135,137,139,141,143,145,147,149,151,153,155,157,159,161,163,165,167,169,171,173,175,177,179,181,183,185,187,189,191,193,195,197,199,201,203,205,207,209,211,213,215,217,219,221,223,225,227,229,231,233,235,237,239,241,243,245,247,249,251,253,255,253,251,249,247,245,243,241,239,237,235,233,231,229,227,225,223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6,4,2};
#define _Wt(x)	(uint16_t)pgm_read_byte(&Wt[(uint8_t)((x)>>8)])
#endif								// -------------------------------------------------------
#endif								// -------------------------------------------------------

ISR(TIMER0_COMPA_vect){									// Ts(32usまたは64us)毎に
	if(btnAcntr>0)	btnAcntr--;						// 8msのチャタリング防止
	if(btnBcntr>0)	btnBcntr--;						// 8msのチャタリング防止
	if(c5<C5)	c5++;												// 0.5秒カウンタ
	if(PIcntr>0){													// 20240314 ピッ!
		PIcntr--;
		if(++cPI==CPI)	cPI=0;							// 440Hz
		if(cPI<CPI/2)		Va=0x0fff;					// 周期の前半(ピッの音量はここで調整)
		else						Va=0;								// 後半は0
		OCR1A=Va>>8;												// Vaを出力(0-255)
		OCR1B=~(Va>>8);											// -Vaを出力
	}else if(S==3 && btnAcntr==0 && btnBcntr==0){	// 20240314 演奏の条件追加
#if WAVEFORM==0				// --- 減衰矩形波 ----------------------------------------
		if(--cntr==0){											// 32分音符長(62.5ms*120/tempo)
			cntr=CNT;
			if(--len1==0){
				note1=_Part1(imelo, p1++);	len1=_len(imelo, note1>>5);	note1&=0x1f;
				if(note1==0)	S=4;							// part1終了!!
				else{
					c1=0;	C1=(_Cnt(imelo, note1)+64)>>7;	// C1=Cnt/128 周期は32us*C1
					v1=V1;												// 初めは最大振幅
					cdecay=8000/Ts;								// 8ms毎に包絡を減衰
				}
			}
			if(--len2==0){
				note2=_Part2(imelo, p2++);	len2=_len(imelo, note2>>5);	note2&=0x1f;
				if(note2==0)	p2--;							// 0ならPart2の演奏終了
				else{
					c2=0;	C2=(_Cnt(imelo, note2)+64)>>7;	// C2=Cnt/128 周期は32us*C2
					v2=V2;												// 初めは最大振幅
				}
			}
			if(--len3==0){
				note3=_Part3(imelo, p3++);	len3=_len(imelo, note3>>5);	note3&=0x1f;
				if(note3==0)	p3--;							// 0ならPart3の演奏終了
				else{
					c3=0;	C3=(_Cnt(imelo, note3)+64)>>7;	// C3=Cnt/128 周期は32us*C3
					v3=V3;												// 初めは最大振幅
				}
			}
		}
		Vb=Va=0;
		if(++cdecay==480/Ts)	cdecay=0;			// 480us周期
		if(note1>1){												// part1が休符でなければ
			if(++c1==C1)		c1=0;							// 周期は32us*C1
			if(c1<(C1>>1))	Va=v1;						// 周期の前半は振幅Va=v1
			else if(cdecay==0)	v1-=(v1>>8);	// 480us毎にv1*=(255/256)
		}
		if(note2>1){												// part2が休符でなければ
			if(++c2==C2)		c2=0;							// 周期は32us*C2
			if(c2<(C2>>1))	Vb=v2;						// 周期の前半は振幅Vb=v2
			else if(cdecay==0)	v2-=(v2>>8);	// 480us毎にv2*=(255/256)
		}
		if(note3>1){												// part3が休符でなければ
			if(++c3==C3)		c3=0;							// 周期は32us*C3
			if(c3<(C3>>1))	Vb+=v3;						// 周期の前半は振幅Vb+=v3
			else if(cdecay==0)	v3-=(v3>>8);	// 480us毎にv3*=(255/256)
		}
		OCR1A=(Va+128)>>8;									// Vaを出力(0-255)
#if NTRACK==1
		OCR1B=~((Va+128)>>8);								// -Vaを出力
#elif NTRACK==2 || NTRACK==3
		OCR1B=~((Vb+128)>>8);								// -Vbを出力
#endif

#else									// --- 減衰正弦波または減衰正弦波(伴奏は三角波) ----------

#if NTRACK==1					// --- TRACK数1 ------------------------------------------
		if(--cntr==0){											// 32分音符長(62.5ms*120/tempo)
			cntr=CNT;
			if(--len1==0){
				note1=_Part1(imelo, p1++);	len1=_len(imelo, note1>>5);	note1&=0x1f;
				if(note1==0)	S=4;							// part1終了!!
				else{														// sw1: 波形テーブルのステップ幅
					sw1=_SW(imelo, note1);	E1=0;	// アタックタイム後にE1=0x8000
				}
				len1a=Ca;												// アタックタイム 4ms
			}
		}
		if(note1==0)			v1=v3=0;					// 終了
		else if(note1==1)	v1=v3=255;				// 休符
		else{																// part1の振幅を加算
			if(len1a>0){	len1a--;	E1+=0x8000/Ca;}		// アタックタイム後にE1=0x8000
			E=0x8000-E1;
			v1=_Ws(pW1+=sw1)+_Ws(pW1+E);			// 0-510
#if WAVEFORM==1				// --- 正弦波で伴奏 --------------------------------------
			v3=_Ws(pW3+=(sw1+10))+_Ws(pW3+E);	// 0-510 わずか周波数をずらす(ビブラート効果)
#elif WAVEFORM==2			// --- 三角波で伴奏 --------------------------------------
			v3=_Wt(pW3+=(sw1+10))+_Wt(pW3+E);	// 0-510 わずか周波数をずらす(ビブラート効果)
#endif								// -------------------------------------------------------
		}
		v=v1+((v1+v3)>>1);									// 0-1020
		OCR1A=v>>2;	OCR1B=(1023-v)>>2;			// 0-255
		if(--cdecay==0){	cdecay=8000/Ts;	E1-=(E1>>5);	}	// 8ms毎に減衰(31/32)

#elif NTRACK==2	// --- TRACK数2 ------------------------------------------------
		if(--cntr==0){											// 32分音符長(62.5ms*120/tempo)
			cntr=CNT;
			if(--len1==0){
				note1=_Part1(imelo, p1++);	len1=_len(imelo, note1>>5);	note1&=0x1f;
				if(note1==0)	S=4;							// part1終了!!
				else{														// sw1: 波形テーブルのステップ幅
					sw1=_SW(imelo, note1);	E1=0;	// アタックタイム後にE1=0x8000
				}
				len1a=Ca;												// アタックタイム 4ms
			}
			if(--len2==0){
				note2=_Part2(imelo, p2++);	len2=_len(imelo, note2>>5);	note2&=0x1f;
				if(note2==0)	p2--;							// Part2終了!!
				else{														// sw2: 波形テーブルのステップ幅
					sw2=_SW(imelo, note2);	E2=0;	// アタックタイム後にE2=0x8000
				}
				len2a=Ca;												// アタックタイム 4ms
			}
		}
		if(note1==0)			v1=v3=0;					// 終了
		else if(note1==1)	v1=v3=255;				// 休符
		else{																				// part1の振幅を加算
			if(len1a>0){	len1a--;	E1+=0x8000/Ca;}		// アタックタイム後にE1=0x8000
			E=0x8000-E1;
			v1=_Ws(pW1+=sw1)+_Ws(pW1+E);			// 0-510
			v3=_Ws(pW3+=(sw1+10))+_Ws(pW3+E);	// 0-510 わずか周波数をずらす(ビブラート効果)
		}
		if(note2==0)			v2=0;							// 終了
		else if(note2==1)	v2=255;						// 休符
		else{																				// part2の振幅を加算
			if(len2a>0){	len2a--;	E2+=0x8000/Ca;}		// アタックタイム後にE2=0x8000
			E=0x8000-E2;
#if WAVEFORM==1				// --- 正弦波で伴奏 --------------------------------------
			v2=_Ws(pW2+=sw2)+_Ws(pW2+E);			// 0-510
#elif WAVEFORM==2			// --- 三角波で伴奏 --------------------------------------
			v2=_Wt(pW2+=sw2)+_Wt(pW2+E);			// 0-510
#endif								// -------------------------------------------------------
		}
		v=v1+((v2+v3)>>1);									// 0-1020
		OCR1A=v>>2;	OCR1B=(1023-v)>>2;			// 0-255
		if(--cdecay==0){cdecay=8000/Ts;	E1-=(E1>>5); E2-=(E2>>5);}// 8ms毎に減衰(31/32)

#elif NTRACK==3				// --- TRACK数3 ------------------------------------------
		if(--cntr==0){											// 32分音符長(62.5ms*120/tempo)
			cntr=CNT;
			if(--len1==0){
				note1=_Part1(imelo, p1++);	len1=_len(imelo, note1>>5);	note1&=0x1f;
				if(note1==0)	S=4;							// part1終了!!
				else{														// sw1: 波形テーブルのステップ幅
					sw1=_SW(imelo, note1);	E1=0;	// アタックタイム後にE1=0x8000
				}
				len1a=Ca;												// アタックタイム 4ms
			}
			if(--len2==0){
				note2=_Part2(imelo, p2++);	len2=_len(imelo, note2>>5);	note2&=0x1f;
				if(note2==0)	p2--;							// 0ならPart2の演奏終了
				else{														// sw2: 波形テーブルのステップ幅
					sw2=_SW(imelo, note2);	E2=0;	// アタックタイム後にE2=0x8000
				}
				len2a=Ca;												// アタックタイム 4ms
			}
			if(--len3==0){
				note3=_Part3(imelo, p3++);	len3=_len(imelo, note3>>5);	note3&=0x1f;
				if(note3==0)	p3--;							// 0ならPart3の演奏終了
				else{														// sw3: 波形テーブルのステップ幅
					sw3=_SW(imelo, note3);	E3=0;	// アタックタイム後にE3=0x8000
				}
				len3a=Ca;												// アタックタイム 4ms
			}
		}
		if(note1==0)			v1=0;							// 終了
		else if(note1==1)	v1=255;						// 休符
		else{																				// part1の振幅を加算
			if(len1a>0){	len1a--;	E1+=0x8000/Ca;}		// アタックタイム後にE1=0x8000
			E=0x8000-E1;
			v1=_Ws(pW1+=sw1)+_Ws(pW1+E);			// 0-510
		}
		if(note2==0)			v2=0;							// 終了
		else if(note2==1)	v2=255;						// 休符
		else{																// part2の振幅を加算
			if(len2a>0){	len2a--;	E2+=0x8000/Ca;}		// アタックタイム後にE2=0x8000
			E=0x8000-E2;
#if WAVEFORM==1				// --- 正弦波で伴奏 --------------------------------------
			v2=_Ws(pW2+=sw2)+_Ws(pW2+E);			// 0-510
#elif WAVEFORM==2			// --- 三角波で伴奏 --------------------------------------
			v2=_Wt(pW2+=sw2)+_Wt(pW2+E);			// 0-510
#endif								// -------------------------------------------------------
		}
		if(note3==0)			v3=0;							// 終了
		else if(note3==1)	v3=255;						// 休符
		else{																				// part3の振幅を加算
			if(len3a>0){	len3a--;	E3+=0x8000/Ca;}		// アタックタイム後にE3=0x8000
			E=0x8000-E3;
#if WAVEFORM==1				// --- 正弦波で伴奏 --------------------------------------
			v3=_Ws(pW3+=sw3)+_Ws(pW3+E);			// 0-510
#elif WAVEFORM==2			// --- 三角波で伴奏 --------------------------------------
			v3=_Wt(pW3+=sw3)+_Wt(pW3+E);			// 0-510
#endif								// -------------------------------------------------------
		}
		v=v1+((v2+v3)>>1);													// 0-1020
		OCR1A=v>>2;	OCR1B=(1023-v)>>2;							// 0-255
		if(--cdecay==0){														// 8ms毎に減衰(31/32)
			cdecay=8000/Ts;	E1-=(E1>>5); E2-=(E2>>5); E3-=(E3>>5);
		}
#endif								// -------------------------------------------------------
#endif								// -------------------------------------------------------
	}
//	OCR0A=Ts*F_CPU/8/1000000;								// 64(F_CPU=16MHz)/32(F_CPU=8MHz)
}

#if defined(__MegaX8__)||defined(__TinyX313__)	// -----------------------------
ISR(INT0_vect){	btnAcntr=Th/Ts;	GIMSK=0;	c5=0;	}				// レベル割込みを禁止
ISR(INT1_vect){	btnBcntr=Th/Ts;	GIMSK=0;				}				// レベル割込みを禁止
#elif defined(__TinyX5__)	// ---------------------------------------------------
ISR(PCINT0_vect){														// btnA または btnB が変化
	if(digitalRead(btnApin)==LOW)				btnAcntr=Th/Ts;		// 8ms
	else if(digitalRead(btnBpin)==LOW)	btnBcntr=Th/Ts;		// 8ms
}
#endif										// ---------------------------------------------------

void setup(){
	pinMode(btnApin, INPUT_PULLUP);	pinMode(btnBpin, INPUT_PULLUP);
	pinMode(Bzz1pin, OUTPUT);	pinMode(Bzz2pin, OUTPUT);
#if defined(__MegaX8__)||defined(__TinyX313__)	// --- 8 bit Fast PWM ----------
											// fPWM=16MHz/256=62.5kHz T=16us(UNO/NANO)
											// fPWM=8MHz/256=31.25kHz T=32us(tiny2313(A)/4313, mega328P)
	TCCR1A=0xa1;											// if(TCNT1==1)						OC1A=1;
	TCCR1B=0x09;											// else if(TCNT1==OCR1A)	OC1A=0;
																		// if(TCNT1==1)						OC1B=1;
																		// else if(TCNT1==OCR1B)	OC1B=0;
#elif defined(__TinyX5__)		// --- 8 bit Fast PWM(fPWM=64/256=250kHz) ----------
	PLLCSR=0x06;											// fPWM=64MHz/256=250kHz T=4us(tiny45/85)
	TCCR1=0x51;												// if(TCNT1==1)						OC1A=1;
																		// else if(TCNT1==OCR1A)	OC1A=0;
	GTCCR=0x50;												// if(TCNT1==1)						OC1B=1;
																		// else if(TCNT1==OCR1B)	OC1B=0;
	OCR1A=OCR1B=128;	OCR1C=255;			// fPWM=64000/(255+1)=250kHz
	PCMSK=(1<<PCINT2)|(1<<PCINT0);		// btnA | btnB PinChange int
#endif											// -------------------------------------------------
	TIMSK=(1<<OCIE0A);
	TCCR0A=0x02;	TCCR0B=0x02;				// TC0: CTC, F_CPU/8
	OCR0A=Ts*F_CPU/8/1000000;			// 64(F_CPU=16MHz,Ts=32us)/64(F_CPU=8MHz,Ts=64us)
	sei();
	S=3;	beginMelo();								// 電源投入時に1曲目を演奏
}

void beginMelo(){										// 電子オルゴールimelo曲目を演奏
	CNT=62500UL/Ts*120/tempo[imelo];	// テンポ120の四分音符(長さ8)は0.5秒
	cntr=1;														// 32分音符長(62.5ms*120/tempo)
	p1=p2=p3=0;												// ポインタ先頭
#if WAVEFORM==1 || WAVEFORM==2
	pW1=pW2=pW3=0;										// ポインタ先頭
#endif											// -------------------------------------------------
	note1=note2=note3=0;	len1=len2=len3=1;	// 初期ディレイ 0.5秒(無音)
}

void pi(){	PIcntr=62500UL/Ts*2;	cPI=0;}	// 20240314 ピッ! 0.125秒

void loop(){
	if(digitalRead(btnApin)==LOW)				btnAcntr=Th/Ts;	// チャタリング防止
	else if(digitalRead(btnBpin)==LOW)	btnBcntr=Th/Ts;	// チャタリング防止
	if(S==0){
		if(btnAcntr>0){				S=1;	imelo=0;	c5=0;	}			// btnAがオンで1曲目
		else if(btnBcntr>0){	S=1;	imelo=(NMELO>1?1:0);}	// btnBがオンで2曲目(あれば)
		else{			// スリープ ******************************************************
			pinMode(Bzz1pin,INPUT);		pinMode(Bzz2pin,INPUT);
#if defined(__MegaX8__)||defined(__TinyX313__)	// -----------------------------
			GIMSK=(1<<INT0)|(1<<INT1);			// SLEEP中のINT0とINT1を許可
#elif defined(__TinyX5__)		// -------------------------------------------------
			GIMSK=(1<<PCIE);								// SLEEP中のPCINTを許可
#endif											// -------------------------------------------------
			set_sleep_mode(SLEEP_MODE_PWR_DOWN);
			sleep_mode();										// SE=1, sleep, SE=0
			GIMSK=0;												// ボタンの割込み禁止
			pinMode(Bzz1pin,OUTPUT);	pinMode(Bzz2pin,OUTPUT);
		}					// スリープ ******************************************************
	}else if(S==1){																	// (btnA/Bがオン)
		if(btnAcntr==0 && btnBcntr==0){	S=2;	c5=0;	}	// btnA/Bオフになったら
		else if(btnAcntr>0 && c5==C5){								// btnA長押しなら
			if(imelo<NMELO-1){	imelo++;	c5=0;	pi();}	// 次の曲(あれば)
		}
	}else if(S==2){																	// (btnA/Bがオフ)
		if(btnAcntr>0){																// btnAオンになったら(連打)
			if(imelo<NMELO-1){	S=1;	imelo++;	c5=0;	}	// 次の曲(あれば)
		}else if(c5==C5){		S=3;	beginMelo();	}			// オフで0.5秒後にimeloを再生!!
	}else if(S==3){																	// 再生中!!
		if(btnAcntr>0 || btnBcntr>0)		S=4;					// btnA/Bがオンで停止
	}else if(S==4){
		if(btnAcntr==0 && btnBcntr==0)	S=0;					// btnA/BがオフでS=0
	}
}
