// 音声メッセージ付電子オルゴール MyMelo2PLUS(録音音声の再生機能を追加)
// https://koyama.verse.jp/elecraft/mymelo/
// 20230616-20251020 koyama88@cameo.plala.or.jp
// (特徴)
// (1) MyMelo2 Plus は、私だけの電子オルゴール MyMelo2 に録音音声を挿入する機能を追加したもので、「音声メッセージ付 電子オルゴール」を作ることができます。オルゴールの音はピアノの音に似たきれいな音色です(減衰正弦波)。
// (2) 音声メッセージはオルゴールのメロディーの任意の場所に挿入できます。複数の音声を別々の場所で再生することもできます。
// (3) 複数の曲を登録し、ボタン操作で再生する曲を選択できます。
// (4) ArduinoUNO/NANO, Digispark, ATtiny85, ATtiny4313, ATmega328P などで動作します。
// (5) 挿入できる音声メッセージの長さは使用するマイコンにより異なり、ATtiny85では中品質で1.3秒、低品質で2.6秒程度です。
//
// (データの初期化)
// (1) はじめに mymelo2plus.wsf をダブルクリックして音声データファイル(mymelo2plusフォルダのmyvoice.h)と楽譜データファイル(mymelo2plusフォルダのmymelo2.h)を初期化します。
//
// (音声データの作成)
// (1) Audacityなどのソフトを使い、パソコンで音声を録音し、wavファイルとして保存します。スマホで録音したファイルをFFMPEGなどのソフトを使って変換することもできます。
// (2) 録音音声ファイル(〇〇.wav)を mymelo2plus.wsf にドラッグ＆ドロップすると、音声データファイルに音声データが記録されます。
// (3) 複数の音声ファイル(〇〇.wav)を用意し、(2)の操作を繰り返すと、音声データファイルに複数の音声を登録できます。ただし、記録できる時間は「合計で」最大1.3秒程度(中品質)または2.6秒程度(低品質)です。
// (4) 音声を変更した場合は、データの初期化からやり直します。
//
// (楽譜データの作成)
// (1) 「テキスト音楽サクラ」(またはメモ帳などのテキストエディタ)で「ドドソソララソー」のように楽譜ファイル(〇〇.mml)を作成します(日本語コードはシフトJIS)。
//		使える音はRA1_～SI8およびREST(休符)の87個から最大31種類です。録音音声の箇所は「n0」「n1」...のように記載します。
//		使える音符はL1_(付点全音符)～L32(32分音符)の11個から最大8種類です。
// (2) 楽譜ファイル(〇〇.mml)を mymelo2plus.wsf にドラッグ＆ドロップすると、楽譜データファイルに楽譜データが記録されます。
// (3) (2)の操作を繰り返すと、楽譜データファイルに複数の楽譜を登録できます。
// (4) 楽譜を変更した場合はデータの初期化からやり直します。
//
// (プログラムの書き込み)
// ArduinoIDEを起動し、mymelo2plusフォルダの mymelo2plus.ino を開いて書き込みます。
//
// (使い方)
// 電源を投入すると、録音メッセージ付のオルゴールが再生されます。外部スイッチをつなぐと、スイッチを押したときに再生されます。スイッチの連打または長押しで2曲目以降が再生されます。

#include <avr/sleep.h>
#include "mymelo2.h"						// 楽譜データ
#include "myvoice.h"						// 録音音声データ

#if F_CPU>=16000000
#define	Ts	32			// サンプリング周期 Ts=32us
#define CPI	71			// 1000000/440/Ts ピッ(440Hz)のカウント
#else
#define	Ts	64			// クロックが遅いと処理が間に合わないのでTs=64usにした
#define CPI	35			// 1000000/440/Ts ピッ(440Hz)のカウント
#endif

#define	Tv	125			// サンプリング周期 Tv=125us
#define Th	8000		// チャタリング防止のためのbtnA, btnBのホールドタイム 8ms
#define C5	(500000/Ts)	// 0.5sカウント
#define Ca	(4096/Ts)		// アタックタイム(4ms)カウント

#if defined(ARDUINO_AVR_UNO)||defined(__AVR_ATmega168__)||defined(__AVR_ATmega328P__)||defined(__AVR_ATmega328__)
#define __MegaX8__
#define GIMSK	EIMSK
#define TIMSK	TIMSK0
#define btnApin	2		// INT0/PCINT18
#define btnBpin	3		// INT1/PCINT19
#define Bzz1pin	9		// OC1A
#define Bzz2pin	10	// OC1B
#elif defined(ARDUINO_AVR_ATTINYX313)
#define __TinyX313__
#define btnApin	4		// INT0/PCINT13
#define btnBpin	5		// INT1/PCINT14
#define Bzz1pin	12	// OC1A
#define Bzz2pin	13	// OC1B
#elif defined(ARDUINO_AVR_ATTINYX5)||defined(__AVR_ATtiny25__)||defined(__AVR_ATtiny45__)||defined(__AVR_ATtiny85__)||defined(ARDUINO_AVR_DIGISPARK)
#define __TinyX5__
#define btnApin 2		// INT0/PCINT2
#define btnBpin 0		// PCINT0
#define Bzz1pin	1		// OC1A
#define Bzz2pin	4		// OC1B
#endif								// -------------------------------------------------------
#define _Voice(i,p)	pgm_read_byte(&((char*)pgm_read_ptr(&Voice[i]))[p])
#define _len(i,n)	pgm_read_byte(&((uint8_t*)pgm_read_ptr(&pLen[i]))[n])
#define _Part1(i,p)	pgm_read_byte(&((uint8_t*)pgm_read_ptr(&pPart1[i]))[p])
#define _Part2(i,p)	pgm_read_byte(&((uint8_t*)pgm_read_ptr(&pPart2[i]))[p])
#define _Part3(i,p)	pgm_read_byte(&((uint8_t*)pgm_read_ptr(&pPart3[i]))[p])
volatile uint8_t btnAcntr=0, btnBcntr=0;	// チャタリング防止カウンタ 32*250=8ms
volatile uint16_t c5=0;					// 0.5秒カウンタ(volatile必要!!)
uint8_t S;											// ボタンの状態

#if defined(MYVOICE)	// -------------------------------------------------------
// Original Code for PIC by Rodger Richey, 1-9-96
// This ADPCM routines were obtained from the Interactive Multimedia Association's
// Reference ADPCM algorithm.  This algorithm was first implemented by Intel/DVI.
#if CODESIZE==4				// -------------------------------------------------------
char IndexTable[]={-1,-1,-1,-1,2,4,6,8};	// Table of index changes
#elif CODESIZE==2			// -------------------------------------------------------
char IndexTable[]={-1,2};									// Table of index changes
#endif								// -------------------------------------------------------
const int16_t StepSizeTable[] PROGMEM={		// Quantizer step size lookup table
	7, 8, 9, 10, 11, 12, 13, 14, 16, 17,
	19, 21, 23, 25, 28, 31, 34, 37, 41, 45,
	50, 55, 60, 66, 73, 80, 88, 97, 107, 118,
	130, 143, 157, 173, 190, 209, 230, 253, 279, 307,
	337, 371, 408, 449, 494, 544, 598, 658, 724, 796,
	876, 963, 1060, 1166, 1282, 1411, 1552, 1707, 1878, 2066,
	2272, 2499, 2749, 3024, 3327, 3660, 4026, 4428, 4871, 5358,
	5894, 6484, 7132, 7845, 8630, 9493, 10442, 11487, 12635, 13899,
	15289, 16818, 18500, 20350, 22385, 24623, 27086, 29794, 32767
};
int16_t lastsample=0;									// Output of ADPCM predictor
int8_t lastindex=0;										// Quantizer step size index
uint8_t code;													// dual 4-bits / quad 2-bits ADPCM code
uint8_t j=0;													// upper 4-bit / lower 4-bit of code
uint8_t ivoice=0;											// Voice[0]を再生
uint16_t p=voicesize[ivoice];					// 音声データのポインタ(初期値は終了状態)

int16_t ADPCMDecoder(uint8_t code){		// code: dual 4-bits / quad 2-bits ADPCM code
	int16_t predsample=lastsample;			// Restore previous values of predicted sample
	int8_t index=lastindex;							// and quantizer step size index
	int16_t diffq;											// Dequantized predicted difference
// Find quantizer step size from lookup table using index
	int16_t step=pgm_read_word(&StepSizeTable[index]);	// Quantizer step size
// Inverse quantize the ADPCM code into a difference using the quantizer step size
#if CODESIZE==4				// -------------------------------------------------------
	diffq=step>>3;
	if(code&4)	diffq+=step;
	if(code&2)	diffq+=step>>1;
	if(code&1)	diffq+=step>>2;
// Add the difference to the predicted sample
	if(code&8){
#elif CODESIZE==2			// -------------------------------------------------------
	diffq=step>>1;
	if(code&1)	diffq+=step;
// Add the difference to the predicted sample
	if(code&2){
#endif								// -------------------------------------------------------
// Check for overflow of the new predicted sample
		if(predsample<0 && predsample<diffq-32768)	predsample=-32768;
		else																				predsample-=diffq;
	}else{
		if(predsample>=0 && diffq>32767-predsample)	predsample=32767;
		else																				predsample+=diffq;
	}
// Find new quantizer step size by adding the old index and a table lookup
// using the ADPCM code
#if CODESIZE==4				// -------------------------------------------------------
	index+=IndexTable[code&0x07];
#elif CODESIZE==2			// -------------------------------------------------------
	index+=IndexTable[code&0x01];
#endif								// -------------------------------------------------------
// Check for overflow of index the new quantizer step size
	if(index<0)				index=0;
	else if(index>88)	index=88;
// Save predicted sample for next iteration and quantizer step size index
	lastsample=predsample;	lastindex=index;
	return predsample;									// Return the new speech sample
}
#endif								// -------------------------------------------------------

uint8_t imelo=0;								// 曲番号(初期値は0)
uint8_t cdecay;									// 8ms毎に包絡を減衰
uint16_t CNT, cntr;							// Ts*(1953*120/tempo)=62.5ms
uint16_t p1, p2, p3;						// 楽譜part1～part3のポインタ
uint8_t note1,note2,note3;			// 楽譜データDO|L4
uint8_t len1, len2, len3;				// 音符の長さ
uint16_t v,v1,v2,v3;						// 振幅とその合算値
uint16_t PIcntr=0;							// 20240314 ピッの長さ
uint8_t cPI=0;									// 20240314 ピッのカウント

// *** SWの表は Ts=32us のものなので「*(Ts/32)」***
#define _SW(i,n)	pgm_read_word(&((uint16_t*)pgm_read_ptr(&pSW[i]))[n])*(Ts/32)
uint8_t len1a=0,len2a=0,len3a=0;			// アタックタイム 4ms
uint16_t E,E1,E2,E3;									// 包絡
uint16_t pW1,pW2,pW3, sw1,sw2,sw3;		// 波形ポインタとサンプリングステップ幅
// 正弦波の波形テーブル(値:0～255, データ数:256)
const uint8_t Ws[] PROGMEM={0,0,0,0,1,1,1,2,2,3,4,5,5,6,7,9,10,11,12,14,15,17,18,20,21,23,25,27,29,31,33,35,37,40,42,44,47,49,52,54,57,59,62,65,67,70,73,76,79,82,85,88,90,93,97,100,103,106,109,112,115,118,121,124,127,131,134,137,140,143,146,149,152,155,158,162,165,167,170,173,176,179,182,185,188,190,193,196,198,201,203,206,208,211,213,215,218,220,222,224,226,228,230,232,234,235,237,238,240,241,243,244,245,246,248,249,250,250,251,252,253,253,254,254,254,255,255,255,255,255,255,255,254,254,254,253,253,252,251,250,250,249,248,246,245,244,243,241,240,238,237,235,234,232,230,228,226,224,222,220,218,215,213,211,208,206,203,201,198,196,193,190,188,185,182,179,176,173,170,167,165,162,158,155,152,149,146,143,140,137,134,131,128,124,121,118,115,112,109,106,103,100,97,93,90,88,85,82,79,76,73,70,67,65,62,59,57,54,52,49,47,44,42,40,37,35,33,31,29,27,25,23,21,20,18,17,15,14,12,11,10,9,7,6,5,5,4,3,2,2,1,1,1,0,0,0};
#define _Ws(x)	(uint16_t)pgm_read_byte(&Ws[(uint8_t)((x)>>8)])

ISR(TIMER0_COMPA_vect){									// Ts(32usまたは64us)毎に
	if(btnAcntr>0)	btnAcntr--;						// 8msのチャタリング防止
	if(btnBcntr>0)	btnBcntr--;						// 8msのチャタリング防止
	if(c5<C5)	c5++;												// 0.5秒カウンタ
	if(PIcntr>0){													// 20240314 ピッ!
		PIcntr--;
		if(++cPI==CPI)	cPI=0;							// 440Hz
		if(cPI<CPI/2)		v=0x0fff;						// 周期の前半(ピッの音量はここで調整)
		else						v=0;								// 後半は0
		OCR1A=v>>8;													// v1を出力(0-255)
		OCR1B=~(v>>8);											// -v1を出力
	}else if(S==3){												// 再生中なら
#if defined(MYVOICE)	// --- MYVOICE -------------------------------------------
		if(p<voicesize[ivoice]){						// 音声再生中なら ************************
			int16_t val;
#if CODESIZE==4				// -------------------------------------------------------
			if(j==0){
				code=_Voice(ivoice, p);
				val=ADPCMDecoder(code>>4);	// upper 4-bits -> 16-bits
 			}else{
				val=ADPCMDecoder(code);			// lower 4-bits -> 16-bits
				if(++p==voicesize[ivoice]){	// 再生終了なら
					OCR0A=Ts*F_CPU/8/1000000;	// 64(F_CPU=16MHz,Ts=32us)/48(F_CPU=8MHz,Ts=48us)
					cntr=1;	len1=len2=len3=1;	// すぐに次のデータ
				}
			}
			j++;	j&=0x01;
#elif CODESIZE==2			// -------------------------------------------------------
			if(j==0){
				code=_Voice(ivoice, p);
				val=ADPCMDecoder(code>>6);	// bit7-6 -> 16-bits
			}else if(j==1){
				val=ADPCMDecoder(code>>4);	// bit5-4 -> 16-bits
 			}else if(j==2){
				val=ADPCMDecoder(code>>2);	// bit3-2 -> 16-bits
 			}else{
				val=ADPCMDecoder(code);			// bit1-0 -> 16-bits
				if(++p==voicesize[ivoice]){	// 再生終了なら
					OCR0A=Ts*F_CPU/8/1000000;	// 64(F_CPU=16MHz,Ts=32us)/48(F_CPU=8MHz,Ts=48us)
					cntr=1;	len1=len2=len3=1;	// すぐに次のデータ
				}
		 	}
			j++;	j&=0x03;
#endif								// --- CODESIZE ------------------------------------------
			OCR1A=128+(val>>8);	OCR1B=128-(val>>8);
		}else{	// 音声再生中でなければ ********************************************
#endif								// --- MYVOICE -------------------------------------------

#if NTRACK==1					// --- TRACK数1 ------------------------------------------
			if(--cntr==0){												// 32分音符長(62.5ms*120/tempo)
				cntr=CNT;
				if(--len1==0){
					note1=_Part1(imelo, p1++);	len1=_len(imelo, note1>>5);	note1&=0x1f;
					if(note1==0)	S=4;								// part1終了!!
#if defined(MYVOICE)	// --- MYVOICE -------------------------------------------
					else if(note1<=NVOICE){						// 音声再生なら
#if Tv*F_CPU/8/1000000 > 255
						OCR0A=255;											// 255(F_CPU=16.5MHz)
#else
						OCR0A=Tv*F_CPU/8/1000000;				// 250(F_CPU=16MHz)/125(F_CPU=8MHz)
#endif
						ivoice=note1-1;									// 再生する音声番号 0～
						p=0; lastsample=0; lastindex=0;	// 音声サンプル先頭から
					}
#endif								// -------------------------------------------------------
					else{															// sw1: 波形テーブルのステップ幅
						sw1=_SW(imelo, note1);	E1=0;		// アタックタイム後にE1=0x8000
					}
					len1a=Ca;													// アタックタイム 4ms
				}
			}
			if(note1==0)							v1=v3=0;		// 終了
			else if(note1==NVOICE+1)	v1=v3=255;	// 休符
			else if(note1>NVOICE+1){							// part1の振幅を加算
				if(len1a>0){	len1a--;	E1+=0x8000/Ca;}	// アタックタイム後にE1=0x8000
				E=0x8000-E1;
				v1=_Ws(pW1+=sw1)+_Ws(pW1+E);				// 0-510
				v3=_Ws(pW3+=(sw1+10))+_Ws(pW3+E);		// 0-510 わずか周波数をずらす(ビブラート効果)
			}
			v=v1+((v1+v3)>>1);										// 0-1020
			OCR1A=v>>2;	OCR1B=(1023-v)>>2;				// 0-255
			if(--cdecay==0){	cdecay=8000/Ts;	E1-=(E1>>5);	}	// 8ms毎に減衰(31/32)

#elif NTRACK==2				// --- TRACK数2 ----------------------------------------
			if(--cntr==0){												// 32分音符長(62.5ms*120/tempo)
				cntr=CNT;
				if(--len1==0){
					note1=_Part1(imelo, p1++);	len1=_len(imelo, note1>>5);	note1&=0x1f;
					if(note1==0)	S=4;								// part1終了!!
#if defined(MYVOICE)	// --- MYVOICE -------------------------------------------
					else if(note1<=NVOICE){						// 音声再生なら
#if Tv*F_CPU/8/1000000 > 255
						OCR0A=255;											// 255(F_CPU=16.5MHz)
#else
						OCR0A=Tv*F_CPU/8/1000000;				// 250(F_CPU=16MHz)/125(F_CPU=8MHz)
#endif
						ivoice=note1-1;									// 再生する音声番号 0～
						p=0; lastsample=0; lastindex=0;	// 音声サンプル先頭から
					}
#endif								// -------------------------------------------------------
					else{															// sw1: 波形テーブルのステップ幅
						sw1=_SW(imelo, note1);	E1=0;		// アタックタイム後にE1=0x8000
					}
					len1a=Ca;													// アタックタイム 4ms
				}
				if(--len2==0){
					note2=_Part2(imelo, p2++);	len2=_len(imelo, note2>>5);	note2&=0x1f;
					if(note2==0)	p2--;								// Part2終了!!
					else if(note2<=NVOICE){	}					// 音声再生なら何もしない
					else{															// sw2: 波形テーブルのステップ幅
						sw2=_SW(imelo, note2);	E2=0;		// アタックタイム後にE2=0x8000
					}
					len2a=Ca;													// アタックタイム 4ms
				}
			}
			if(note1==0)							v1=v3=0;		// 終了
			else if(note1==NVOICE+1)	v1=v3=255;	// 休符
			else if(note1>NVOICE+1){							// part1の振幅を加算
			if(len1a>0){len1a--;	E1+=0x8000/Ca;}	// アタックタイム後にE1=0x8000
				E=0x8000-E1;
				v1=_Ws(pW1+=sw1)+_Ws(pW1+E);				// 0-510
				v3=_Ws(pW3+=(sw1+10))+_Ws(pW3+E);		// 0-510 わずか周波数をずらす(ビブラート効果)
			}
			if(note2==0)							v2=0;				// 終了
			else if(note2==NVOICE+1)	v2=255;			// 休符
			else if(note2>NVOICE+1){							// part2の振幅を加算
				if(len2a>0){len2a--;E2+=0x8000/Ca;}	// アタックタイム後にE2=0x8000
				E=0x8000-E2;
				v2=_Ws(pW2+=sw2)+_Ws(pW2+E);				// 0-510(正弦波で伴奏)
			}
			v=v1+((v2+v3)>>1);										// 0-1020
			OCR1A=v>>2;	OCR1B=(1023-v)>>2;				// 0-255
			if(--cdecay==0){cdecay=8000/Ts;	E1-=(E1>>5); E2-=(E2>>5);}// 8ms毎に減衰(31/32)

#elif NTRACK==3				// --- TRACK数3 ------------------------------------------
			if(--cntr==0){												// 32分音符長(62.5ms*120/tempo)
				cntr=CNT;
				if(--len1==0){
					note1=_Part1(imelo, p1++);	len1=_len(imelo, note1>>5);	note1&=0x1f;
					if(note1==0)	S=4;								// part1終了!!
#if defined(MYVOICE)	// --- MYVOICE -------------------------------------------
					else if(note1<=NVOICE){						// 音声再生なら
#if Tv*F_CPU/8/1000000 > 255
						OCR0A=255;											// 255(F_CPU=16.5MHz)
#else
						OCR0A=Tv*F_CPU/8/1000000;				// 250(F_CPU=16MHz)/125(F_CPU=8MHz)
#endif
						ivoice=note1-1;									// 再生する音声番号 0～
						p=0; lastsample=0; lastindex=0;	// 音声サンプル先頭から
					}
#endif								// -------------------------------------------------------
					else{															// sw1: 波形テーブルのステップ幅
						sw1=_SW(imelo, note1);	E1=0;		// アタックタイム後にE1=0x8000
					}
					len1a=Ca;													// アタックタイム 4ms
				}

				if(--len2==0){
					note2=_Part2(imelo, p2++);	len2=_len(imelo, note2>>5);	note2&=0x1f;
					if(note2==0)	p2--;								// Part2終了!!
					else if(note2<=NVOICE){	}					// 音声再生なら何もしない
					else{															// sw2: 波形テーブルのステップ幅
						sw2=_SW(imelo, note2);	E2=0;		// アタックタイム後にE2=0x8000
					}
					len2a=Ca;													// アタックタイム 4ms
				}
				if(--len3==0){
					note3=_Part3(imelo, p3++);	len3=_len(imelo, note3>>5);	note3&=0x1f;
					if(note3==0)	p3--;								// Part3終了!!
					else if(note3<=NVOICE){	}					// 音声再生なら何もしない
					else{															// sw3: 波形テーブルのステップ幅
						sw3=_SW(imelo, note3);	E3=0;		// アタックタイム後にE3=0x8000
					}
					len3a=Ca;													// アタックタイム 4ms
				}
			}
			if(note1==0)							v1=0;				// 終了
			else if(note1==NVOICE+1)	v1=255;			// 休符
			else if(note1>NVOICE+1){							// part1の振幅を加算
				if(len1a>0){	len1a--;	E1+=0x8000/Ca;}	// アタックタイム後にE1=0x8000
				E=0x8000-E1;
				v1=_Ws(pW1+=sw1)+_Ws(pW1+E);				// 0-510
			}
			if(note2==0)							v2=0;				// 終了
			else if(note2==NVOICE+1)	v2=255;			// 休符
			else if(note2>NVOICE+1){							// part2の振幅を加算
				if(len2a>0){	len2a--;	E2+=0x8000/Ca;}	// アタックタイム後にE2=0x8000
				E=0x8000-E2;
				v2=_Ws(pW2+=sw2)+_Ws(pW2+E);				// 0-510
			}
			if(note3==0)							v3=0;				// 終了
			else if(note3==NVOICE+1)	v3=255;			// 休符
			else if(note3>NVOICE+1){									// part3の振幅を加算
				if(len3a>0){	len3a--;	E3+=0x8000/Ca;}	// アタックタイム後にE3=0x8000
				E=0x8000-E3;
				v3=_Ws(pW3+=sw3)+_Ws(pW3+E);				// 0-510
			}
			v=v1+((v2+v3)>>1);										// 0-1020
			OCR1A=v>>2;	OCR1B=(1023-v)>>2;				// 0-255
			if(--cdecay==0){											// 8ms毎に減衰(31/32)
				cdecay=8000/Ts;	E1-=(E1>>5); E2-=(E2>>5); E3-=(E3>>5);
			}
#endif								// -------------------------------------------------------
#if defined(MYVOICE)	// --- MYVOICE -------------------------------------------
#if defined(__TinyX5__)			// -------------------------------------------------
//		OCR0A=Ts*F_CPU/8/1000000;							// 64(F_CPU=16MHz)/32(F_CPU=8MHz)
#endif											// -------------------------------------------------
		}									// 		if(p<voicesize[ivoice]){}else{	// 音声再生中でなければ
#else									// -------------------------------------------------------
//		OCR0A=Ts*F_CPU/8/1000000;							// 64(F_CPU=16MHz)/32(F_CPU=8MHz)
#endif								// -------------------------------------------------------
	}										// if(nplay>0){
}

#if defined(__MegaX8__)||defined(__TinyX313__)	// -----------------------------
ISR(INT0_vect){	btnAcntr=Th/Ts;	GIMSK=0;	c5=0;	}				// レベル割込みを禁止
ISR(INT1_vect){	btnBcntr=Th/Ts;	GIMSK=0;				}				// レベル割込みを禁止
#elif defined(__TinyX5__)	// ---------------------------------------------------
ISR(PCINT0_vect){														// btnA または btnB が変化
	if(digitalRead(btnApin)==LOW)				btnAcntr=Th/Ts;		// 8ms
	else if(digitalRead(btnBpin)==LOW)	btnBcntr=Th/Ts;		// 8ms
}
#endif										// ---------------------------------------------------

void setup(){
	pinMode(btnApin, INPUT_PULLUP);	pinMode(btnBpin, INPUT_PULLUP);
	pinMode(Bzz1pin, OUTPUT);	pinMode(Bzz2pin, OUTPUT);
#if defined(__MegaX8__)||defined(__TinyX313__)	// --- 8 bit Fast PWM ----------
	TCCR1A=0xa1;											// if(TCNT1==1)					OC1A=1;
	TCCR1B=0x09;											// else if(TCNT1==OCR1A)OC1A=0;
																		// if(TCNT1==1)					OC1B=1;
																		// else if(TCNT1==OCR1B)OC1B=0;
#elif defined(__TinyX5__)		// --- 8 bit Fast PWM ------------------------------
	PLLCSR=0x06;											// TC1...250kHz PWM
	TCCR1=0x51;												// if(TCNT1==1)						OC1A=1;
																		// else if(TCNT1==OCR1A)	OC1A=0;
	GTCCR=0x50;												// if(TCNT1==1)						OC1B=1;
																		// else if(TCNT1==OCR1B)	OC1B=0;
	OCR1A=OCR1B=128;	OCR1C=255;			// fPWM=64000/(255+1)=250kHz
	PCMSK=(1<<PCINT2)|(1<<PCINT0);		// btnA | btnB PinChange int
#endif											// -------------------------------------------------
	TIMSK=(1<<OCIE0A);
	TCCR0A=0x02;	TCCR0B=0x02;				// TC0: CTC, F_CPU/8
	OCR0A=Ts*F_CPU/8/1000000;			// 64(F_CPU=16MHz,Ts=32us)/64(F_CPU=8MHz,Ts=64us)
	sei();
	S=3;	beginMelo();								// 電源投入時に1曲目を演奏
}

void beginMelo(){										// 電子オルゴールimelo曲目を演奏
	OCR0A=Ts*F_CPU/8/1000000;			// 64(F_CPU=16MHz,Ts=32us)/64(F_CPU=8MHz,Ts=64us)
	CNT=62500UL/Ts*120/tempo[imelo];	// テンポ120の四分音符(長さ8)は0.5秒
	cntr=CNT;													// 32分音符長(62.5ms*120/tempo)
	p1=p2=p3=0;	pW1=pW2=pW3=0;				// ポインタ先頭
	note1=note2=note3=0;	len1=len2=len3=1;	// 初期ディレイ 0.5秒(無音)
}

void pi(){	PIcntr=62500UL/Ts*2;	cPI=0;}	// 20240314 ピッ! 0.125秒

void loop(){
	if(digitalRead(btnApin)==LOW)				btnAcntr=Th/Ts;	// チャタリング防止
	else if(digitalRead(btnBpin)==LOW)	btnBcntr=Th/Ts;	// チャタリング防止
	if(S==0){																			// (btnA/Bがオフ)
		if(btnAcntr>0){				S=1;	imelo=0;	c5=0;	}			// btnAがオンで1曲目
		else if(btnBcntr>0){	S=1;	imelo=(NMELO>1?1:0);}	// btnBがオンで2曲目(あれば)
		else{			// スリープ ******************************************************
			pinMode(Bzz1pin,INPUT);		pinMode(Bzz2pin,INPUT);
#if defined(__MegaX8__)||defined(__TinyX313__)	// -----------------------------
			GIMSK=(1<<INT0)|(1<<INT1);			// SLEEP中のINT0とINT1を許可
#elif defined(__TinyX5__)		// -------------------------------------------------
			GIMSK=(1<<PCIE);								// SLEEP中のPCINTを許可
#endif											// -------------------------------------------------
			set_sleep_mode(SLEEP_MODE_PWR_DOWN);
			sleep_mode();										// SE=1, sleep, SE=0
			GIMSK=0;												// ボタンの割込み禁止
			pinMode(Bzz1pin,OUTPUT);	pinMode(Bzz2pin,OUTPUT);
		}					// スリープ ******************************************************
	}else if(S==1){																	// (btnA/Bがオン)
		if(btnAcntr==0 && btnBcntr==0){	S=2;	c5=0;	}	// btnA/Bオフになったら
		else if(btnAcntr>0 && c5==C5){								// btnA長押しなら
			if(imelo<NMELO-1){	imelo++;	c5=0;	pi();}	// 次の曲(あれば)
		}
	}else if(S==2){																	// (btnA/Bがオフ)
		if(btnAcntr>0){																// btnAオンになったら(連打)
			if(imelo<NMELO-1){	S=1;	imelo++;	c5=0;	}	// 次の曲(あれば)
		}else if(c5==C5){		S=3;	beginMelo();	}			// オフで0.5秒後にimeloを再生!!
	}else if(S==3){																	// 再生中!!
		if(btnAcntr>0 || btnBcntr>0)		S=4;					// btnA/Bがオンで停止
	}else if(S==4){
		if(btnAcntr==0 && btnBcntr==0)	S=0;					// btnA/BがオフでS=0
	}
}