// 音声メッセージプレイヤー MyVoice
// https://koyama.verse.jp/elecraft/myvoice
// 20230616-20251020 koyama88@cameo.plala.or.jp
// (特徴)
// (1) 短い音声メッセージ(録音音声)を再生できます。
// (2) 複数の音声を登録し、ボタン操作で再生する音声を選択できます。
// (3) ArduinoUNO/NANO, Digispark, ATtiny85, ATtiny4313, ATmega328Pなどで動作します。
// (4) 記録できる音声メッセージの長さは使用するマイコンにより異なり、ATtiny85では中品質で1.5秒、低品質で3秒程度です。
//
// (音声データの作成)
// (1) Audacityなどのソフトを使い、パソコンで音声を録音し、wavファイルとして保存します。スマホで録音したファイルをFFMPEGなどのソフトを使って変換することもできます。
// (2) はじめに myvoice.wsf をダブルクリックして音声データファイル(myvoice.h)を初期化します。
// (3) 録音音声ファイル(〇〇.wav)を myvoice.wsf にドラッグ＆ドロップすると、音声データファイルに音声データが記録されます。
// (4) (3)の操作を繰り返すと、音声データファイルに複数の音声を登録できます。ただし、記録できる時間は「合計で」マイコンの容量の範囲となります。
// (5) 音声を変更した場合は、(2)の初期化からやり直します。
//
// (プログラムの書き込み)
// ArduinoIDEを起動し、myvoice.ino を開いて書き込みます。
//
// (使い方)
// 電源を投入すると、録音メッセージが再生されます。外部スイッチをつなぐと、スイッチを押したときに再生されます。スイッチの連打または長押しで2つ目以降の音声を再生できます。

// Original Code for PIC by Rodger Richey, 1-9-96
// This ADPCM routines were obtained from the Interactive Multimedia Association's
// Reference ADPCM algorithm.  This algorithm was first implemented by Intel/DVI.

#include <avr/sleep.h>
#include "myvoice.h"

#define	Ts	125			// サンプリング周期 Ts=125us
#define Th	8000		// チャタリング防止のためのbtnA, btnBのホールドタイム 8ms
#define C5	(500000/Ts)	// 0.5sカウント

#if defined(ARDUINO_AVR_UNO)||defined(__AVR_ATmega168__)||defined(__AVR_ATmega328P__)||defined(__AVR_ATmega328__)
#define __MegaX8__
#define GIMSK	EIMSK
#define TIMSK	TIMSK0
#define btnApin	2		// INT0/PCINT18
#define btnBpin	3		// INT1/PCINT19
#define Bzz1pin	9		// OC1A
#define Bzz2pin	10	// OC1B
#elif defined(ARDUINO_AVR_ATTINYX313)
#define __TinyX313__
#define btnApin	4		// INT0/PCINT13
#define btnBpin	5		// INT1/PCINT14
#define Bzz1pin	12	// OC1A
#define Bzz2pin	13	// OC1B
#elif defined(ARDUINO_AVR_ATTINYX5)||defined(__AVR_ATtiny25__)||defined(__AVR_ATtiny45__)||defined(__AVR_ATtiny85__)||defined(ARDUINO_AVR_DIGISPARK)
#define __TinyX5__
#define btnApin 2		// INT0/PCINT2
#define btnBpin 0		// PCINT0
#define Bzz1pin	1		// OC1A
#define Bzz2pin	4		// OC1B
#endif								// -------------------------------------------------------

#define _Voice(i,p)	pgm_read_byte(&((char*)pgm_read_ptr(&Voice[i]))[p])

volatile uint8_t btnAcntr=0, btnBcntr=0;	// チャタリング防止カウンタ 125*64=8ms
volatile uint16_t c5=0;					// 0.5秒カウンタ(volatile必要!!)
uint8_t S;											// ボタンの状態

#if CODESIZE==4		// ---------------------------------------------------------
char IndexTable[]={-1,-1,-1,-1,2,4,6,8};	// Table of index changes
#elif CODESIZE==2	// ---------------------------------------------------------
char IndexTable[]={-1,2};									// Table of index changes
#endif						// ---------------------------------------------------------
const int16_t StepSizeTable[] PROGMEM={		// Quantizer step size lookup table
    7, 8, 9, 10, 11, 12, 13, 14, 16, 17,
    19, 21, 23, 25, 28, 31, 34, 37, 41, 45,
    50, 55, 60, 66, 73, 80, 88, 97, 107, 118,
    130, 143, 157, 173, 190, 209, 230, 253, 279, 307,
    337, 371, 408, 449, 494, 544, 598, 658, 724, 796,
    876, 963, 1060, 1166, 1282, 1411, 1552, 1707, 1878, 2066,
    2272, 2499, 2749, 3024, 3327, 3660, 4026, 4428, 4871, 5358,
    5894, 6484, 7132, 7845, 8630, 9493, 10442, 11487, 12635, 13899,
    15289, 16818, 18500, 20350, 22385, 24623, 27086, 29794, 32767
};
int16_t lastsample=0;									// Output of ADPCM predictor
int8_t lastindex=0;										// Quantizer step size index
uint8_t code;													// dual 4-bits ADPCM code
uint8_t j=0;													// upper 4-bit / lower 4-bit of code
uint8_t ivoice=0;											// 再生する音声の番号(Voice[0]を再生)
uint16_t p=0;													// 音声データのポインタ(初期値は先頭)

int16_t ADPCMDecoder(uint8_t code){		// code: dual 4-bits / quad 2-bits ADPCM code
	int16_t predsample=lastsample;			// Restore previous values of predicted sample
	int8_t index=lastindex;							// and quantizer step size index
	int16_t diffq;											// Dequantized predicted difference
// Find quantizer step size from lookup table using index
	int16_t step=pgm_read_word(&StepSizeTable[index]);	// Quantizer step size
// Inverse quantize the ADPCM code into a difference using the quantizer step size
#if CODESIZE==4		// ---------------------------------------------------------
	diffq=step>>3;
	if(code&4)	diffq+=step;
	if(code&2)	diffq+=step>>1;
	if(code&1)	diffq+=step>>2;
// Add the difference to the predicted sample
	if(code&8){
#elif CODESIZE==2	// ---------------------------------------------------------
	diffq=step>>1;
	if(code&1)	diffq+=step;
// Add the difference to the predicted sample
	if(code&2){
#endif						// ---------------------------------------------------------
// Check for overflow of the new predicted sample
		if(predsample<0 && predsample<diffq-32768)	predsample=-32768;
		else																				predsample-=diffq;
	}else{
		if(predsample>=0 && diffq>32767-predsample)	predsample=32767;
		else																				predsample+=diffq;
	}
// Find new quantizer step size by adding the old index and a table lookup
// using the ADPCM code
#if CODESIZE==4		// ---------------------------------------------------------
	index+=IndexTable[code&0x07];
#elif CODESIZE==2	// ---------------------------------------------------------
	index+=IndexTable[code&0x01];
#endif						// ---------------------------------------------------------
// Check for overflow of index the new quantizer step size
	if(index<0)				index=0;
	else if(index>88)	index=88;
// Save predicted sample for next iteration and quantizer step size index
	lastsample=predsample;	lastindex=index;
	return predsample;									// Return the new speech sample
}

#define CPI	18									// 1000000/440/Ts ピッ(440Hz)のカウント
uint16_t v;											// 振幅とその合算値
uint16_t PIcntr=0;							// 20240314 ピッの長さ
uint8_t cPI=0;									// 20240314 ピッのカウント
ISR(TIMER0_COMPA_vect){									// Ts(125us)毎に
	if(btnAcntr>0)	btnAcntr--;						// 8msのチャタリング防止
	if(btnBcntr>0)	btnBcntr--;						// 8msのチャタリング防止
	if(c5<C5)	c5++;												// 0.5秒カウンタ
	if(PIcntr>0){													// 20240314 ピッ!
		PIcntr--;
		if(++cPI==CPI)	cPI=0;							// 440Hz
		if(cPI<CPI/2)		v=0x0fff;						// 周期の前半(ピッの音量はここで調整)
		else						v=0;								// 後半は0
		OCR1A=v>>8;													// vを出力(0-255)
		OCR1B=~(v>>8);											// -vを出力
	}else if(S==3 && p<voicesize[ivoice]){// 再生中なら
		int16_t val;
#if CODESIZE==4		// ---------------------------------------------------------
		if(j==0){
			code=_Voice(ivoice, p);
			val=ADPCMDecoder(code>>4);			// upper 4-bits -> 16-bits
 		}else{
			val=ADPCMDecoder(code);					// lower 4-bits -> 16-bits
			if(++p==voicesize[ivoice])	S=4;// 再生終了
		}
		j++;	j&=0x01;
#elif CODESIZE==2	// ---------------------------------------------------------
		if(j==0){
			code=_Voice(ivoice, p);
			val=ADPCMDecoder(code>>6);			// bit7-6 -> 16-bits
		}else if(j==1){
			val=ADPCMDecoder(code>>4);			// bit5-4 -> 16-bits
 		}else if(j==2){
			val=ADPCMDecoder(code>>2);			// bit3-2 -> 16-bits
 		}else{
			val=ADPCMDecoder(code);					// bit1-0 -> 16-bits
			if(++p==voicesize[ivoice])	S=4;// 再生終了
	 	}
		j++;	j&=0x03;
#endif						// ---------------------------------------------------------
		OCR1A=128+(val/256);	OCR1B=128-(val/256);
	}
}

#if defined(__MegaX8__)||defined(__TinyX313__)	// -----------------------------
ISR(INT0_vect){	btnAcntr=Th/Ts;	GIMSK=0;		c5=0;	}	// レベル割込みをここで禁止
ISR(INT1_vect){	btnBcntr=Th/Ts;	GIMSK=0;					}	// レベル割込みをここで禁止
#elif defined(__TinyX5__)		// -------------------------------------------------
ISR(PCINT0_vect){														// btnA または btnB が変化
	if(digitalRead(btnApin)==LOW)				btnAcntr=Th/Ts;	// 8ms
	else if(digitalRead(btnBpin)==LOW)	btnBcntr=Th/Ts;	// 8ms
}
#endif											// -------------------------------------------------

void setup(){
	pinMode(btnApin, INPUT_PULLUP);	pinMode(btnBpin, INPUT_PULLUP);
	pinMode(Bzz1pin, OUTPUT);	pinMode(Bzz2pin, OUTPUT);
#if defined(__MegaX8__)			// --- 8 bit Fast PWM ------------------------------
	TCCR1A=0xa1;												// if(TCNT1==1)					OC1A=1;
	TCCR1B=0x09;												// else if(TCNT1==OCR1A)OC1A=0;
																			// if(TCNT1==1)					OC1B=1;
																			// else if(TCNT1==OCR1B)OC1B=0;
#elif defined(__TinyX313__)	// --- 8 bit Fast PWM ------------------------------
	TCCR1A=0xa1;												// if(TCNT1L==OCR1AL)		OC1A=0;	|~~|_|
	TCCR1B=0x09;												// else if(TCNT1L==255)	OC1A=1;
																			// if(TCNT1L==OCR1BL)		OC1B=1;	|__|~|
																			// else if(TCNT1L==255)	OC1B=0;
																			// Timer1: 8 bit Fast PWM, CK/1
#elif defined(__TinyX5__)		// --- 8 bit Fast PWM ------------------------------
	PLLCSR=0x06;												// TC1...250kHz PWM
	TCCR1=0x51;													// if(TCNT1==1)						OC1A=1;
																			// else if(TCNT1==OCR1A)	OC1A=0;
	GTCCR=0x50;			 										// if(TCNT1==1)						OC1B=1;
																			// else if(TCNT1==OCR1B)	OC1B=0;
	OCR1A=OCR1B=128;	OCR1C=255;				// fPWM=64000/(255+1)=250kHz
	PCMSK=(1<<PCINT2)|(1<<PCINT0);			// btnA | btnB PinChange int
#endif											// -------------------------------------------------
	TIMSK=(1<<OCIE0A);
	TCCR0A=0x02;	TCCR0B=0x02;					// CTC, F_CPU/8, 割込み周期125us
#if Ts*F_CPU/8/1000000 > 255
	OCR0A=255;													// 255(F_CPU=16.5MHz)
#else
	OCR0A=Ts*F_CPU/8/1000000;						// 250(F_CPU=16MHz)/125(F_CPU=8MHz)
#endif
	sei();
}

void beginVoice(){										// ivoiceの録音音声を再生
	p=0; j=0;	lastsample=0; lastindex=0;
}

void pi(){	PIcntr=62500UL/Ts*2;	cPI=0;}	// 20240314 ピッ! 0.125秒

void loop(){
	if(digitalRead(btnApin)==LOW)				btnAcntr=Th/Ts;		// チャタリング防止
	else if(digitalRead(btnBpin)==LOW)	btnBcntr=Th/Ts;		// チャタリング防止
	if(S==0){																							// (btnA/Bがオフ)
		if(btnAcntr>0){				S=1;	ivoice=0;	c5=0;	}				// btnAがオンで1番目の音声
		else if(btnBcntr>0){	S=1;	ivoice=(NVOICE>1?1:0);}	// btnBがオンで2番目の音声(あれば)
		else{			// スリープ ******************************************************
			pinMode(Bzz1pin,INPUT);		pinMode(Bzz2pin,INPUT);
#if defined(__MegaX8__)||defined(__TinyX313__)	// -----------------------------
			GIMSK=(1<<INT0)|(1<<INT1);			// SLEEP中のINT0とINT1を許可
#elif defined(__TinyX5__)		// -------------------------------------------------
			GIMSK=(1<<PCIE);								// SLEEP中のPCINTを許可
#endif											// -------------------------------------------------
			set_sleep_mode(SLEEP_MODE_PWR_DOWN);
			sleep_mode();										// SE=1, sleep, SE=0
			GIMSK=0;												// ボタンの割込み禁止
			pinMode(Bzz1pin,OUTPUT);	pinMode(Bzz2pin,OUTPUT);
		}					// スリープ ******************************************************
	}else if(S==1){																	// (btnA/Bがオン)
		if(btnAcntr==0 && btnBcntr==0){	S=2;	c5=0;	}	// btnA/Bオフになったら
		else if(btnAcntr>0 && c5==C5){								// btnA長押しなら
			if(ivoice<NVOICE-1){ivoice++;	c5=0;	pi();}	// 次の音声(あれば)
		}
	}else if(S==2){																	// (btnA/Bがオフ)
		if(btnAcntr>0){																// btnAオンになったら(連打)
			if(ivoice<NVOICE-1){S=1;	ivoice++;	c5=0;	}	// 次の音声(あれば)
		}else if(c5==C5){	S=3;	beginVoice();	}				// オフで0.5秒後にivoiceを再生!!
	}else if(S==3){																	// 再生中!!
		if(btnAcntr>0 || btnBcntr>0)		S=4;					// btnA/Bがオンで停止
	}else if(S==4){
		if(btnAcntr==0 && btnBcntr==0)	S=0;					// btnA/BがオフでS=0
	}
}