// WebIO2 20250402 koyama88(at)cameo.plala.or.jp
// WebIO2は、MIDI通信を用いてブラウザからIO制御する仕組みです(詳細は下記)
// https://koyama.verse.jp/elecraft/webio/
// デバイス...以下のArduinoデバイスに、スケッチ(WebIO2.ino)を書き込んでおきます
//		USB接続: Leonardo, ProMicro, Wio, XIAOSAMD, XIAORP
//		Bluetooth接続: XIAOC3, XIAOS3, ATOMC3lite, ATOMS3lite
// ブラウザ
//		USB接続: WebMidiAPI対応ブラウザ(Windows版Chrome, Edgeなど)
//		Bluetooth接続: WebMidiAPIおよびWebBluetooth対応ブラウザ
//									(Windows版Chrome, Edge, Android版Chromeなど)
// 特徴:
//		・Webページのプログラム(HTML+Javascript)でさまざまな実世界の制御ができます
//		・Webページのプログラムをサーバに置いて共同利用できます
//				制御対象は「ブラウザが動作しているパソコンやスマホに接続したデバイス」で、
//				インターネット経由でデバイスを操作できるというわけではありません。
// 例:
//		・ブラウザ画面に表示する温湿度計
//				大きなモニタを使ってイベント会場などで熱中症対策に利用できます
//				https://koyama.verse.jp/co2checker/
//		・ブラウザ画面に表示するCO2濃度モニタ
//				大きなモニタを使ってイベント会場などで換気の目安を表示できます
//				https://koyama.verse.jp/co2checker/
//		・ブラウザ画面で操作する赤外線リモコン
//				リモコンボタンを押せない重度障害者もパソコンが操作できればリモコンを
//				操作できます
//				https://koyama.verse.jp/elecraft/irremocon/
//		・ものづくり教育
//				ブラウザの操作でLEDのオンオフ、スイッチでブラウザ画面を操作、
//				センサの値をブラウザ画面に表示など、電子工作の導入教育に利用できます
//				https://koyama.verse.jp/elecraft/bbxiao/webio2xiao.html

#if defined(__AVR_ATmega168__)||defined(__AVR_ATmega328P__)||defined(__AVR_ATmega328__)
#define __MEGAX8__
#elif defined(__AVR_ATtiny2313__) || defined(__AVR_ATtiny4313__)
#define __TINYX313__
#elif defined(__AVR_ATtiny25__)||defined(__AVR_ATtiny45__)||defined(__AVR_ATtiny85__)
#define __TINYX5__
#elif defined(ARDUINO_AVR_LEONARDO)
#define _LEONARDO
#define _USBMIDI
#elif defined(ARDUINO_AVR_MICRO)
#define _PROMICRO
#define _USBMIDI
#elif defined(ARDUINO_WIO_TERMINAL)
#define _WIO								// __SAMD51__ also defined
#define _USBMIDI
#elif defined(ARDUINO_SEEED_XIAO_M0)
#define _XIAOM0							// __SAMD21__ also defined
#define _USBMIDI
#elif defined(ARDUINO_ARCH_RP2040)
#define _XIAORP
#define _USBMIDI
#elif defined(ARDUINO_XIAO_ESP32C3)
#define _XIAOC3							// ESP32 also defined
#define _BLEMIDI
#elif defined(ARDUINO_XIAO_ESP32S3)
#define _XIAOS3							// ESP32 also defined
#define _BLEMIDI
#elif defined(ARDUINO_M5STACK_ATOM)
#define _ATOMC3							// ESP32 also defined
#define _BLEMIDI
#elif defined(ARDUINO_M5STACK_ATOMS3)
#define _ATOMS3							// ESP32 also defined
#define _BLEMIDI
#elif defined(ARDUINO_M5STACK_NANOC6)
#define _NANOC6							// ESP32 also defined
#define _BLEMIDI
#endif
#include <Arduino.h>

#if defined(_LEONARDO)||defined(_PROMICRO)
#else
#include <IRremote.hpp>
IRData sIRData;
#endif
// 温度センサを発熱の大きいマイコンなどと同じケースに収めた場合に補正(実測に基づく)
#define SCDOFFSET	2.2			// SCD30の温度オフセット(M5STACK)
#define SHTOFFSET	1.1			// SHT31の温度オフセット(M5STACK)
#define WIOOFFSET	2.2			// SCD30の温度オフセット(WIO)
#include <Wire.h>
#include "Adafruit_SHT31.h"
Adafruit_SHT31 sht31;
#include "SparkFun_SCD30_Arduino_Library.h"
SCD30 scd30;
#define Ttx		10000			// 少なくとも10秒毎に送信
#define MAXPIN		64		// ピンの最大数(バッファ容量の関係)
#define TEMPpin		MAXPIN-1	// 仮想ピン SHT31/SCD30
#define HUMIpin		MAXPIN-2	// 仮想ピン SHT31/SCD30
#define CO2pin		MAXPIN-3	// 仮想ピン SCD30
#define LMOTORpin	MAXPIN-4	// 仮想ピン 左MOTOR
#define RMOTORpin	MAXPIN-5	// 仮想ピン 右MOTOR
#define tempL	(-45.)		// 温度の最低値(無効データ)
#define AVESIZE		10		// 平均計算のデータ数
// -----------------------------------------------
//	(pinMode:0-13) PROMICRO	SAMD	RP	ESP32
//			INPUT						0			0		0		1
//			OUTPUT					1			1		1		3
//			(PULLUP												4)
//			INPUT_PULLUP		2			2		2		5
//			(PULLDOWN											8)
//			(INPUT_PULLDOWN				3		3		9)
//			(OUTPUT_2MA								4)
//			(OUTPUT_4MA								5)
//			(OUTPUT_8MA								6)
//			(OUTPUT_12MA							7)
//			(OPEN_DRAIN										16)
//			(OUTPUT_OPEN_DRAIN						18)
//			(ANALOG												192)
#define INPUT_ANALOG 		6
#define OUTPUT_ANALOG 	7
#define OUTPUT_TONE			10
#define OUTPUT_NEOPIXEL	11
#define INPUT_IR				12
#define OUTPUT_IR				13
// --- NeoPixel --------------------------------------------
#define NUM_PIXELS	12	// LED数
#include <Adafruit_NeoPixel.h>
Adafruit_NeoPixel *pixels;
// --- Servo -----------------------------------------------
#if defined(_XIAOC3)||defined(_XIAOS3)||defined(_ATOMC3)||defined(_ATOMS3)||defined(_NANOC6)
#else										// *** LEONARDO, PROMICRO, XIAOSAMD, WIO ***
#include <Servo.h>
Servo myservo;
#define TSERVO	300			// SERVOは300msでdetatch
#endif
// ---------------------------------------------------------
#include "pitchToFrequency.h"
// ---------- デバイス別のpin番号 --------------------------
#if defined(_LEONARDO)	// -----------------------------------------------------
#define NPIN	25
bool isSMP=false;				// SMPbegin()でtrue
int SMPt=0;							// SMP duration(20ms単位)
byte SMPy[8];						// SMP data
static const int8_t D[]={
	0,1,2,3,4,5,6,7,8,9,10,11,12,13,-1,-1,-1,-1,18,19,20,21,22,23	};	// D0～のGPIO
static const int8_t A[]={	18,19,20,21,22,23,4,6,8,9,10,12	};				// A0～のGPIO
//					GPIO0				// D0
//							1				// D1
//							2				// D2 / SDA
//							3 			// D3 / SCL
//							4				// D4 / A6
//							5#			// D5
//							6#			// D6 / A7
//							7				// D7
//							8				// D8 / A8
//							9#			// D9 / A9
//							10#			// D10 / A10
//							11			// D11
//							12			// D12 / A11
//							13			// D13, LED_BUILTIN, PIN_LED
#define PIN_LED	13			// D13, LED_BUILTIN, PIN_LED
//							18			// D18 / A0
//							19			// D19 / A1
//							20			// D20 / A2
//							21			// D21 / A3
//							22			// D22 / A4
//							23			// D23 / A5
//							4				// D4 / A6
//							6				// D6 / A7
//							8				// D8 / A8
//							9				// D9 / A9
//							10			// D10 / A10
//							12			// D12 / A11

#elif defined(_PROMICRO)	// ---------------------------------------------------
#define NPIN	22
static const int8_t D[]={	0,1,2,3,4,5,6,7,8,9,10,-1,-1,-1,14,15,16,-1,18,19,20,21	};
																										// D0～のGPIO
static const int8_t A[]={	18,19,20,21,-1,-1,4,6	};	// A0～のGPIO
//					GPIO0				// D0
//							1				// D1
//							2				// D2 / SDA
//							3 			// D3 / SCL
//							4				// D4 / A6
//							5#			// D5
//							6#			// D6 / A7
//							7				// D7
//							8				// D8 / A8
//							9#			// D9 / A9
//							10#			// D10
//							14			// D14
//							15			// D15
//							16			// D16
#define A0			18			// D18 / A0
#define A1			19			// D19 / A1
#define A2			20			// D20 / A2
#define A3			21			// D21 / A3
#define A6			4				// D4 / A6
#define A7			6				// D6 / A7
#define A8			8				// D8 / A8
#define A9			9				// D9 / A9
#define A10			10			// D10 / A10
#elif defined(_WIO)	// ---------------------------------------------------------
#define NPIN	47
static const int8_t D[]={	0,1,2,3,4,5,6,7,8	};	// D0～のGPIO
static const int8_t A[]={	0,1,2,3,4,5,6,7,8	};	// A0～のGPIO
//					GPIO0				// D0 /A0(12bit)
//							1				// D1 /A1
//							2				// D2 /A2
//							3				// D3 /A3
//							4				// D4 /A4
//							5				// D5 /A5
//							6				// D6 /A6
//							7				// D7 /A7
//							8				// D8 /A8
//							9				// D9 /A9, DAC0
//							10			// DAC1
//							12			// BUZZER_CTR/WIO_BUZZER
//							13			// PIN_LED, LED_BUILTIN, PIN_NEOPIXEL
//							14			// WIO_IR(sensor)
//							27			// WIO_LIGHT(light sensor)
//							28			// BUTTON_1/WIO_KEY_A
//							29			// BUTTON_2/WIO_KEY_B
//							30			// BUTTON_3/WIO_KEY_C
//							31			// SWITCH_X/WIO_5S_UP
//							32			// SWITCH_Y/WIO_5S_LEFT
//							33			// SWITCH_Z/WIO_5S_RIGHT
//							34			// SWITCH_B/WIO_5S_DOWN
//							35			// SWITCH_U/WIO_5S_PRESS
//							39			// MIC_INPUT/WIO_MIC
//							45			// SCL
//							46			// SDA
#elif defined(_XIAOM0)	// ----------------------------------------------------
#include <SI4735.h>			// ***** SI4732A *****
SI4735 rx;
#include <patch_init.h> // SSB patch for whole SSBRX initialization string
#define FM	0
#define AM	1
#define USB	2
#define LSB	3
#define CW	4
uint8_t rxMode=FM;
bool isRX=false;				// RXsetup()でtrueに
bool bfoOn=false, ssbLoaded=false;
const uint16_t size_content=sizeof ssb_patch_content; // see patch_init.h
uint16_t rxFreq=0, rxfreq=0;	// 直前のfreq
uint8_t rxStatus;				// 直前のstatus...bit0:PILOT bit1:FM bit2:AM bit3:SSB
uint8_t rxRssi=0;				// 直前のRSSI(0-127)

#define NPIN	14
static const int8_t D[]={	0,1,2,3,4,5,6,7,8,9,10,-1,-1,13	};	// D0～のGPIO
static const int8_t A[]={	0,1,2,3,4,5,6,7,8,9,10	};	// A0～のGPIO
//					GPIO0				// D0 /A0(12bit), DAC0
//							1				// D1#/A1
//							2				// D2#/A2
//							3				// D3#/A3
//							4				// D4#/A4, SDA
//							5 			// D5#/A5, SCL
//							6				// D6#/A6
//							7				// D7#/A7, PIN_IRIN
//							8				// D8#/A8, PIN_IROUT
//							9				// D9#/A9
//							10			// D10#/A10
//							13			// D13#, PIN_LED, LED_BUILTIN
#elif defined(_XIAORP)	// -----------------------------------------------------
#define NPIN	30
static const int8_t D[]={	26,27,28,29,6,7,0,1,2,4,3	};	// D0～のGPIO
static const int8_t A[]={	26,27,28,29	};								// A0～のGPIO
//							GPIO26			// D0 /A0
//									27			// D1#/A1
//									28			// D2#/A2
//									29			// D3#/A3(RP2040)
//									6				// D4#, SDA
//									7 			// D5#, SCL
//									0				// D6
//									1				// D7#, PIN_IRIN
//									2				// D8#, PIN_IROUT
//									4				// D9#
//									3				// D10#
//									17			// PIN_LED, PIN_LED_R, LED_BUILTIN(RP2040)
//									16			// PIN_LED_G(RP2040)
//									25			// PIN_LED_B(RP2040) / PIN_LED, LED_BUILTIN(RP2350)
//									12			// PIN_NEOPIXEL(RP2040)
//	NEOPIXEL_POWER	11
//	SDA							6
//	SCL							7
#elif defined(_XIAOC3)	// -----------------------------------------------------
#define BLENAME	"XIAO C3"
#define NPIN	22
static const int8_t D[]={	2,3,4,5,6,7,21,20,8,9,10	};	// D0～のGPIO
static const int8_t A[]={	2,3,4,5	};										// A0～のGPIO
//					GPIO2				// D0 /A0
//							3				// D1 /A1
//							4				// D2#/A2
//							5				// D3#/A3
//							6				// D4#, SDA
//							7 			// D5#, SCL
//							21			// D6
//							20			// D7#, PIN_IRIN
//							8				// D8#, PIN_IROUT
//							9				// D9#
//							10			// D10#
#elif defined(_XIAOS3)	// -----------------------------------------------------
#define BLENAME	"XIAO S3"
#define NPIN	45
static const int8_t D[]={	1,2,3,4,5,6,43,44,7,8,9	};	// D0～のGPIO
static const int8_t A[]={	1,2,3,4,5,6	};							// A0～のGPIO
//					GPIO2				// D0 /A0
//							3				// D1#/A1
//							4				// D2#/A2
//							5				// D3#/A3
//							6				// D4#, SDA
//							7 			// D5#, SCL
//							21			// D6#
//							20			// D7#, PIN_IRIN
//							8				// D8#/A8, PIN_IROUT
//							9				// D9#/A9
//							10			// D10#/A10
#elif defined(_ATOMC3)	// -----------------------------------------------------
#define BLENAME	"ATOM C3"
#define NPIN	40
static const int8_t D[]={	25,21,22,19,23,33,32,26,27,12,39	};	// D0～のGPIO
static const int8_t A[]={	25,21,22,19,23,33,32,26,27,12,39	};	// A0～のGPIO
//										GPIO25	// DAC1
//												21
//												22
//												19
//												23
//												33
//												35	// ADC1
//												36	// ADC2
#define GROVE1						32	// GROVE1(input), SCL, PIN_IRIN
#define GROVE2						26	// GROVE2(output), SDA, PIN_IROUT, DAC2
#define PIN_NEOPIXEL			27	// PIN_NEOPIXEL, NEOPIXEL_BUILTIN
#define NEOPIXEL_BUILTIN	27	// PIN_NEOPIXEL, NEOPIXEL_BUILTIN
#define PIN_RGBLED				27	// PIN_RGBLED, RGBLED_BUILTIN
#define RGBLED_BUILTIN		27	// PIN_RGBLED, RGBLED_BUILTIN
#define PIN_LED						27	// PIN_LED, LED_BUILTIN
#define LED_BUILTIN				27	// PIN_LED, LED_BUILTIN
#define PIN_IRLED					12	// PIN_IRLED, IRLED_BUILTIN
#define IRLED_BUILTIN			12	// PIN_IRLED, IRLED_BUILTIN
#define PIN_BUTTON				39	// PIN_BUTTON, BUTTON_BUILTIN
#define BUTTON_BUILTIN		39	// PIN_BUTTON, BUTTON_BUILTIN
#elif defined(_ATOMS3)	// -----------------------------------------------------
#define BLENAME	"ATOM S3"
#define NPIN	43
static const int8_t D[]={	38,39,5,6,7,8,1,2,-1,4,41	};	// D0～のGPIO
static const int8_t A[]={	38,39,5,6,7,8,1,2,-1,4,41	};	// A0～のGPIO
//										GPIO38	// SDA
//												39	// SCL
//												5
//												6
//												7		// ADC1
//												8		// ADC2
#define GROVE1						1		// GROVE1(input), (SCL), PIN_IRIN
#define GROVE2						2		// GROVE2(output), (SDA), PIN_IROUT
#define PIN_NEOPIXEL			35	// PIN_NEOPIXEL, NEOPIXEL_BUILTIN
#define NEOPIXEL_BUILTIN	35	// PIN_NEOPIXEL, NEOPIXEL_BUILTIN
#define PIN_RGBLED				35	// PIN_RGBLED, RGBLED_BUILTIN
#define RGBLED_BUILTIN		35	// PIN_RGBLED, RGBLED_BUILTIN
#define PIN_LED						35	// PIN_LED, LED_BUILTIN
#define LED_BUILTIN				35	// PIN_LED, LED_BUILTIN
#define PIN_IRLED					4		// PIN_IRLED, IRLED_BUILTIN
#define IRLED_BUILTIN			4		// PIN_IRLED, IRLED_BUILTIN
#define PIN_BUTTON				41	// PIN_BUTTON, BUTTON_BULTIN
#define BUTTON_BUILTIN		41	// PIN_BUTTON, BUTTON_BULTIN
#elif defined(_NANOC6)	// -----------------------------------------------------
#define BLENAME	"NANO C6"
#define NPIN	21
static const int8_t D[]={	1,2	};	// D0～のGPIO
static const int8_t A[]={	1,2	};	// A0～のGPIO
#define GROVE1						1		// D1, A1, GROVE1(input), SCL, PIN_IRIN
#define GROVE2						2		// D2, A2, GROVE2(output), SDA, PIN_IROUT
#define NEOPIXEL_ENABLE		19	// RGB_LED_PWR_PIN, RGBLED_ENABLE(HIGHでENABLE)
#define NEOPIXEL_BUILTIN	20	// RGB_LED_DATA_PIN, RGBLED_BUILTIN(NEOPIXEL)
#define LED_BUILTIN				7		// BLUE_LED_PIN, LED_BUILTIN
#define IRLED_BUILTIN			3		// IR_TX_PIN, IRLED_BUILTIN
#define BUTTON_BUILTIN		9		// BTN_PIN, BUTTON_BUILTIN
#endif

#if defined(_USBMIDI)		// --- USB接続の場合 ---------------
#include <USB-MIDI.h>
USBMIDI_CREATE_DEFAULT_INSTANCE();
#elif defined(_BLEMIDI)	// --- BLE接続の場合 ---------------
#include <BLEMIDI_Transport.h>
#include <hardware/BLEMIDI_ESP32.h>
BLEMIDI_CREATE_INSTANCE(BLENAME, MIDI)
#endif									// ---------------------------------

// ---------- (I2C) ----------------------------------------
#define SCD 		0x61		// SCD30
#define SHT 		0x44		// SHT31(0x44: address pin to GND, 0x45: Open)
#define BUSBUSY (digitalRead(SCL)==LOW||digitalRead(SDA)==LOW)
#define M1 			0x64
#define M2 			0x60
// ---------------------------------------------------------
bool isConnected=false;
bool isSCD=false;							// SCD30の有無
bool isSHT=false;							// SHT31の有無
int pinStatus[MAXPIN];				// ピンの入出力モード
int PinVal[MAXPIN];						// ピンの入出力値
float tempave, tempbuf[AVESIZE];			// 平均値算出用バッファ
int		humiave, humibuf[AVESIZE];			// 平均値算出用バッファ
int		co2ave,  co2buf[AVESIZE];				// 平均値算出用バッファ
int		pave=0;													// 平均値算出用バッファのポインタ
unsigned long lastTxTemp=0, lastTxHumi=0, lastTxCO2=0;	// 前回送信時刻

int TONEpin=-1, NEOPIXELpin=-1;
byte lastbusy=0;
unsigned long lastMillis, measMillis=0, servoMillis=0;
#if defined(_XIAOM0)			// -------------------------------------------------
unsigned long aliveMillis=0, rssiMillis=0, freqMillis=0;
#endif // --------------------------------------------------------------------
void Hexprint(byte v){		// "0F "の表示
	Serial.print(v>>4, HEX);	Serial.print(v&0x0F, HEX);	Serial.print(" ");
}
void Hexprintln(byte v){	// "0F\n"の表示
	Serial.print(v>>4, HEX);	Serial.println(v&0x0F, HEX);
}
// --- 内部処理 ------------------------------------------------------
void servoWrite(uint8_t pin, uint8_t deg){	// deg: 0-180
#if defined(ESP32) 					// -------------------------------------------------
#elif defined(_XIAOM0)			// -------------------------------------------------
	myservo.attach(pin, 1000, 2000);		// 1000-2000[ms](指定なしだと544-2400[ms])
	myservo.write(deg);
	delay(TSERVO);
	myservo.detach();										// 300ms後にdetatch **********************
#elif defined(_LEONARDO)||defined(_PROMICRO)	// -------------------------------
//int us=map(deg, 0,180, 544,2400);		// 544-2400 [ms]
	int us=map(deg, 0,180, 1000,2000);	// 1000-2000 [ms]
	if(pin==9 || pin==10){
		if(deg==0xFF){										// 停止
			TCCR1B&=0xE0;										// WGM1:00 CS1:000
			if(pin==9)	TCCR1A&=0x3C;				// COM1A:00 WGM1:00
			else				TCCR1A&=0xCC;				// COM1B:00 WGM1:00
		}else{														// 開始
			TCCR1B=TCCR1B&0xE0|0x18|0x02;		// WGM1:11 CS1:010(Ck/8)
			ICR1=40000;											// 20ms周期(0.5us×40000)
			if(pin==9){	OCR1A=us<<1;	TCCR1A=TCCR1A&0x3C|0x82;	}	// COM1A:10 WGM1:10
			else{				OCR1B=us<<1;	TCCR1A=TCCR1A&0xCC|0x22;	}	// COM1B:10 WGM1:10
		}
	}else	for(int t=0; t<300; t+=20){
		digitalWrite(pin, HIGH);	delayMicroseconds(us);
		digitalWrite(pin, LOW);		delayMicroseconds(20000-us);
	}
#endif																// ---------------------------------------
}
int dpin(uint8_t pin){	// D.indexOf(pin)
	for(int i=0;i<sizeof(D);i++)	if(D[i]==pin)	return i;
	return -1;
}
int apin(uint8_t pin){	// A.indexOf(pin)
	for(int i=0;i<sizeof(A);i++)	if(A[i]==pin)	return i;
	return -1;
}
void setPinMode(byte pin, byte mode){	// pin 0-63(GPIOpin), mode 0-13
	if(pinStatus[pin]==mode)	return;		// 既にpinがmodeになっていれば何もしない
	if(pin==TEMPpin){										// 仮想ピン
		if(!isSHT){
			sht31=Adafruit_SHT31();
			isSHT=sht31.begin(SHT);					// SHT31の有無
			if(!isSHT && !isSCD)	isSCD=scd30.begin();
		}
		if((isSHT||isSCD)&&mode==INPUT_ANALOG&&pinStatus[pin]!=mode){
			pinStatus[pin]=mode;	PinVal[pin]=tempL;
		}
	}else if(pin==HUMIpin){							// 仮想ピン
		if(!isSHT){
			sht31=Adafruit_SHT31();
			isSHT=sht31.begin(SHT);					// SHT31の有無
			if(!isSHT && !isSCD)	isSCD=scd30.begin();
		}
		if((isSHT||isSCD)&&mode==INPUT_ANALOG&&pinStatus[pin]!=mode){
			pinStatus[pin]=mode;	PinVal[pin]=-1;
		}
	}else if(pin==CO2pin){							// 仮想ピン
		if(mode==INPUT_ANALOG && !isSCD){
			isSCD=scd30.begin();						// SCD30の有無
			if(isSCD)	scd30.setTemperatureOffset(WIOOFFSET);	// SCD30は内部で温度補正
			if(isSCD){	pinStatus[pin]=mode;	PinVal[pin]=-1;	}
		}
#if defined(_ATOMC3)||defined(_ATOMS3)// ---------------------------------------
	}else if(pin==LED_BUILTIN && mode==OUTPUT){	// LED_BUILTINをNEOPIXEL_BUILTINで代行
		NEOPIXELpin=pin;
		pixels=new Adafruit_NeoPixel(1, pin, NEO_GRB+NEO_KHZ800);
		pixels->begin();
#endif																// ---------------------------------------
	}else{															// 通常のピン(pinはGPIOpin 0-59)
		if(dpin(pin)<0)	return;
		pinStatus[pin]=mode;	PinVal[pin]=-1;
		if(mode==INPUT)									pinMode(pin, INPUT);
		else if(mode==OUTPUT){					pinMode(pin, OUTPUT);		digitalWrite(pin, LOW);}
		else if(mode==INPUT_PULLUP)			pinMode(pin, INPUT_PULLUP);
		else if(mode==INPUT_ANALOG)			pinMode(pin, INPUT);
		else if(mode==OUTPUT_TONE){
			TONEpin=pin;									pinMode(pin, OUTPUT);
		}else if(mode==OUTPUT_NEOPIXEL){
			NEOPIXELpin=pin;
			pixels=new Adafruit_NeoPixel(NUM_PIXELS, pin, NEO_GRB+NEO_KHZ800);
			pixels->begin();
#if defined(_LEONARDO)||defined(_PROMICRO)
#else
		}else if(mode==INPUT_IR){
			pinMode(pin, INPUT);
			IrReceiver.begin(pin);
		}else if(mode==OUTPUT_IR){
			pinMode(pin, OUTPUT);
			IrSender.begin(pin);
#endif
		}
	}
}
// --- 温度・湿度・CO2の測定 -----------------------------------------
void measure(){						// 結果は tempave, humiave, co2ave
	float temp=tempL;				// 無効データ
	int humi=-1, co2=-1;		// 無効データ
	if(isSHT){													// SHT31があれば
		temp=sht31.readTemperature()-SHTOFFSET;	// 温度(補正値)
		if(isnan(temp))	temp=tempL;				// 無効データ
		humi=sht31.readHumidity();				// 湿度
		if(isnan(humi)||humi>100||humi<0)	humi=-1;	// 無効データ
	}
	if(isSCD){													// SCD30があれば
		if(scd30.dataAvailable()){
			co2=scd30.getCO2();							// CO2
			if(co2>=10000||co2<200)	co2=-1;	// 無効データ
			if(!isSHT){											// SHT31がなければSCDの温度・湿度
				temp=scd30.getTemperature();	// 温度(デバイスで補正済)
				humi=scd30.getHumidity();			// 湿度
			}
		}
	}
	tempbuf[pave]=temp;	humibuf[pave]=humi;	co2buf[pave]=co2;
//	Serial.println(String(temp)+","+humi+","+co2);	// 測定値の確認
	int tempcount=0, humicount=0, co2count=0;
	tempave=0.;	humiave=0;	co2ave=0;
	for(int i=0; i<AVESIZE; i++){	// 正常値の平均
		if(tempbuf[i]>tempL){	tempave+=tempbuf[i];	tempcount++;}
		if(humibuf[i]>=0){		humiave+=humibuf[i];	humicount++;}
		if(co2buf[i]>=0){			co2ave+=co2buf[i];		co2count++;	}
	}
	tempave=(tempcount>0? tempave/tempcount:tempL);	// 直近10秒間(5～6個)の平均
	humiave=(humicount>0? humiave/humicount: -1);		// 直近10秒間(5～6個)の平均
	co2ave =(co2count>0?  co2ave/co2count: -1);			// 直近10秒間(5～6個)の平均
	pave=(pave+1)%AVESIZE;							// pave: 0～9
//Serial.println(String(tempave)+","+humiave+","+co2ave);	// 平均値の確認
}
// *** コールバック関数 ********************************************************
#if defined(_BLEMIDI) 		// ---------------------------------------------------
void BLEconnected(){			// BLE接続時のコールバック
	isConnected=true;
	Serial.println("BLE connected");
}
void BLEdisconnected(){		// BLE切断時のコールバック
	isConnected=false;
	Serial.println("BLE disconnected");
}
#endif										// ---------------------------------------------------
// --- 起動時およびSystemReset受信時のコールバック -------------------
void pinReset(){
	Serial.println("MIDI RESET!");
	for(int pin=0; pin<MAXPIN; pin++){	// すべてのピンを未設定にする
		pinStatus[pin]=-1;
		PinVal[pin]=-1;
	}
	for(int i=0; i<sizeof(D); i++){
		if(D[i]>=0)	pinMode(D[i], INPUT);	// 有効なDピンをINPUTにする
	}
	lastTxTemp=0; lastTxHumi=0; lastTxCO2=0;	// 温度等はすぐにデータを送る
	isSHT=false; isSCD=false;
	TONEpin=-1; NEOPIXELpin=-1;
	for(int i=0; i<AVESIZE; i++){				// バッファを無効データに
		tempbuf[i]=tempL;	humibuf[i]=-1;	co2buf[i]=-1;
	}
}
// --- NoteOff受信時のコールバック -----------------------------------
void handleNoteOff(uint8_t ch, uint8_t note, uint8_t velocity, uint16_t tstamp){
	if(TONEpin>=0)	tone(TONEpin, pitchFrequency[note]);	// ブザー音をオン
}
// --- NoteOn受信時のコールバック ------------------------------------
void handleNoteOn(uint8_t ch, uint8_t note, uint8_t velocity, uint16_t tstamp){
	if(TONEpin>=0)	noTone(TONEpin);											// ブザー音をオフ
}

// *** PCからSysExメッセージを受信 *********************************************
void OnMidiSysEx(byte* data, unsigned length){		// SysEx受信時のコールバック
	Serial.print(F("received: ("));	Serial.print(length);	Serial.print(F(" bytes) "));
	for(uint16_t i=0; i<length; i++)	Hexprint(data[i]);	Serial.println();
	if(data[0]==0xF0 && data[1]==0x7D){
		if(data[2]==0x70){							// F0 7D 70
			byte pin=data[3];							// 0ppppppp
			byte val=data[4];							// 0100mmmm / 0000000v
			if(val&0x40){									// F0 7D 70 0ppppppp 0100mmmm
// --- pinMode -------------------------------------------------------
				setPinMode(pin, val&0x0F);	// pin0-127のpinModeを0-15に
				Serial.print("pinMode: ");	Hexprint(pin);	Hexprintln(val&0x0F);
			}else{												// F0 7D 70 0ppppppp 0000000v
// --- digitalWrite ---------------------------------------------------
				if(pinStatus[pin]!=OUTPUT) setPinMode(pin, OUTPUT);	// pin0-127をOUTPUTに
#if defined(_ATOMC3)||defined(_ATOMS3)// ---------------------------------------
				if(pin==LED_BUILTIN){				// LED_BUILTINをNEOPIXEL_BUILTINで代行
					pixels->setPixelColor(0, (val&0x01?0x80:0),0,0);	// valが1で赤点灯
					pixels->show();
				}else
#endif															// ---------------------------------------
				{														// 通常は
					digitalWrite(pin, val&0x01);// pin0-127にvを出力
					Serial.print("digitalWrite: ");	Hexprint(pin);	Hexprintln(val&0x01);
				}
			}
		}else if(data[2]==0x6F){				// F0 7D 6F 0ppppppp 0vvvvvv 0vvvvvvv
// --- analogWrite ---------------------------------------------------
			byte pin=data[3];							// 0ppppppp
			byte val=data[4] | (data[5]<<7);	// 0vvvvvvv(LSB) 0vvvvvvv(MSB)
			if(pinStatus[pin]!=OUTPUT) setPinMode(pin, OUTPUT);	// pin0-31をOUTPUTに
			if(pin==LMOTORpin){						// *** 左モーター ***
				char speed=val;							// -128～127
				byte dir;
				if(speed==0){			dir=3; val=0;	}												// STOP
				else if(speed>0){ dir=1; val=map( speed, 1,127, 6,63);}	// FWD
				else{							dir=2; val=map(-speed, 1,128, 6,63);}	// BWD
				Wire.beginTransmission(M1);
				Wire.write(0x00); Wire.write((val<<2)|dir);
				Wire.endTransmission();
			}else if(pin==RMOTORpin){			// *** 右モーター ***
				char speed=val;							// -128～127
				byte dir;
				if(speed==0){			dir=3; val=0;	}												// STOP
				else if(speed>0){ dir=1; val=map( speed, 1,127, 6,63);}	// FWD
				else{							dir=2; val=map(-speed, 1,128, 6,63);}	// BWD
				Wire.beginTransmission(M2);
				Wire.write(0x00); Wire.write((val<<2)|dir);
				Wire.endTransmission();
			}else	analogWrite(pin, val);
			Serial.print("analogWrite: ");	Hexprint(pin);	Hexprintln(val);
#if defined(_XIAOM0)||defined(_LEONARDO)||defined(_PROMICRO)		// -------------
//		servoWrite(pin, val);					// *** ServoMotor ***
#endif										// ---------------------------------------------------
		}else if(data[2]==0x72){								// NeoPixel: F0 7D 72
// --- NeoPixel ------------------------------------------------------
			byte iPixel=data[3]&0x7F,							// 0LLLLLLL 00000rgb
				R=((data[4]&0x04)<<5)|data[5],			// 0rrrrrrr
				G=((data[4]&0x02)<<6)|data[6],			// 0ggggggg
				B=((data[4]&0x01)<<7)|data[7];			// 0bbbbbbb F7
			Serial.print("NEOPIXELpin: ");	Serial.println(NEOPIXELpin);
			Serial.print("LED: ");					Serial.println(iPixel);
			Serial.print("RGB: ");					Hexprint(R);	Hexprint(G);	Hexprintln(B);
			if(NEOPIXELpin>=0){	pixels->setPixelColor(iPixel, R,G,B);	pixels->show();	}
		}else if(data[2]==0x73){								// IR Remote: F0 7D 73
// --- IRデータの送信指示 --------------------------------------------
#if defined(_LEONARDO)||defined(_PROMICRO)
#else
			sIRData.protocol=(decode_type_t)data[3];						// 0PPPPPPP
			sIRData.address=(data[4]&0x7F) | ((data[5]&0x7F)<<7)// 0aaaaaaa(LSB) 0aaaaaaa
											| ((data[6]&0x03)<<14);							// 000000aa(MSB)
			sIRData.command=(data[7]&0x7F) | ((data[8]&0x7F)<<7)// 0ccccccc(LSB) 0ccccccc
											| ((data[9]&0x03)<<14);							// 000000cc(MSB)
			sIRData.flags=IRDATA_FLAGS_EMPTY;
			IrSender.write(&sIRData, 1);					// *** IR信号発生 ***
			Serial.print("IRsend: ");		Hexprint(sIRData.protocol);
			Hexprint(sIRData.address);	Hexprintln(sIRData.command);
#endif
// --- SMPディスプレイへの表示指示 -----------------------------------
#if defined(_LEONARDO)
		}else if(data[2]==0x74){								// SMP: F0 7D 74
			if(data[3]==1){												// SMPbegin()
				isSMP=true;
				DDRB=0xFF;	PORTB=0;								// PB0-7をOUTPUTにする
				DDRD=0xFF;	PORTD=0;								// PD0-7をOUTPUTにする
			}else if(isSMP && data[3]==2){				// SMPdrive()
				SMPt=data[4]*5;											// t:0-127(0.1秒単位)...SMPtは20ms単位
				SMPy[0]=data[5]&0x0F | (data[6]<<4);	// y1
				SMPy[1]=data[7]&0x0F | (data[8]<<4);	// y2
				SMPy[2]=data[9]&0x0F | (data[10]<<4);	// y3
				SMPy[3]=data[11]&0x0F | (data[12]<<4);// y4
				SMPy[4]=data[13]&0x0F | (data[14]<<4);// y5
				SMPy[5]=data[15]&0x0F | (data[16]<<4);// y6
				SMPy[6]=data[17]&0x0F | (data[18]<<4);// y7
				SMPy[7]=data[19]&0x0F | (data[20]<<4);// y8
			}
#endif
// --- SI4732A の制御 ------------------------------------------------
#if defined(_XIAOM0)	// ---------------------------------------------
		}else if(data[2]==0x75){		// SI4732A: F0 7D 75
			if(data[3]==1){
				if(data[4]==0x00){			// RXreset()
					rx.reset();
				}else if(data[4]==0x01){// RXalive()
					aliveMillis=millis();
				}else{									// RXsetup(pin, mode) ... pin(1-7), mode(0-7)
					rx.setI2CFastModeCustom(100000);
					isRX=rx.getDeviceI2CAddress(3);	// Looks for the I2C bus address and set it.	Returns 0 if error
					rx.setup((data[4]>>4)&0x07, (data[4]>>1)&0x07);	// pin(1-7), mode(0-7)
					aliveMillis=millis();
				}
			}else if(data[3]==0x02){
				if(data[4]<64)				rx.setVolume(data[4]);	// RXsetVolume(0-63)
				else if(data[4]==64)	rx.volumeUp();					// RXvolumeUp()
				else if(data[4]==65)	rx.volumeDown();				// RXvolumeDown()
				else if(data[4]==66)	rx.setAudioMute(true);	// RXsetAudioMute(1)
				else if(data[4]==67)	rx.setAudioMute(false);	// RXsetAudioMute(0)
				else if(data[4]==68)	rx.frequencyUp();				// RXfrequencyUp()
				else if(data[4]==69)	rx.frequencyDown();			// RXfrequencyDown()
				else if(data[4]==70 && (rxMode==0||rxMode==1))	rx.seekStationUp();			// RXseekStationUp()
				else if(data[4]==71 && (rxMode==0||rxMode==1))	rx.seekStationDown();		// RXseekStationDown()
				else if(data[4]==72 && (rxMode==0||rxMode==1))	rx.seekStationProgress(seekFreq,1);	// RXseekStationProgress(1)
				else if(data[4]==73 && (rxMode==0||rxMode==1))	rx.seekStationProgress(seekFreq,0);	// RXseekStationProgress(0)
			}else if(data[3]==0x03){
				if(data[6]&0x04){							// RXsetSSBBfo(-16383 - 16383)
					int16_t freq=data[4]|(data[5]<<7)|((data[6]&0x03)<<14);
					freq-=16383;
					rx.setSSBBfo(freq);
				}else{												// RXsetFrequency(freq)
					uint16_t freq=data[4]|(data[5]<<7)|((data[6]&0x03)<<14);
					rx.setFrequency(freq);
				}
			}else if(data[3]==0x04){
				if(data[4]&0xF0==0x00){				// RXsetAutomaticGainControl(disable, agc)
					uint8_t agc=data[4]&0x3F;		// 0-36
					if(agc==0)	rx.setAutomaticGainControl(1, 0);
					else				rx.setAutomaticGainControl(0,agc);
				}else if(data[4]&0x40){				// RXsetBandwidth(chflt, plflt)
					uint8_t bwidx=data[4]&0x07;	// 0:6kHz 1:4kHz 2:3kHz 3:2kHz 4:1kHz 5:1.8kHz 6:2.5kHz
					rx.setBandwidth(bwidx, ((data[4]&0x20)?1:0));	// AM Power Line Noise Rejection Filter
				}else if(data[4]&0x50){ 			// RXsetSSBAudioBandwidth(idx)
					uint8_t idx=data[4]&0x07;		// 0:1.2kHzLPF 1:2.2kHzLPF 2:3kHzLPF 3:4kHzLPF 4:500HzBPF 5:1kHzBPF
					rx.setSSBAudioBandwidth(idx);
					if(idx==0||idx==4||idx==5)	rx.setSSBSidebandCutoffFilter(0);
					else												rx.setSSBSidebandCutoffFilter(1);
				}
			}else if(data[3]==0x05){				// RXsetAM(minfreq, maxfreq, freq, step)
				uint16_t minfreq=data[4]|(data[5]<<7)|((data[6]&0x03)<<14);
				uint16_t maxfreq=data[7]|(data[8]<<7)|((data[9]&0x03)<<14);
				uint16_t freq=data[10]|(data[11]<<7)|((data[12]&0x03)<<14);
				uint16_t step=data[13];
				rx.setAM(minfreq, maxfreq, freq, step);
				rxMode=AM;	rxStatus=0;
				rxFreq=rxfreq=freq;
				bfoOn=ssbLoaded=false;
			}else if(data[3]==0x06){				// RXsetFM(minfreq, maxfreq, freq, step)
				uint16_t minfreq=data[4]|(data[5]<<7)|((data[6]&0x03)<<14);
				uint16_t maxfreq=data[7]|(data[8]<<7)|((data[9]&0x03)<<14);
				uint16_t freq=data[10]|(data[11]<<7)|((data[12]&0x03)<<14);
				uint16_t step=data[13];
//			rx.setTuneFrequencyAntennaCapacitor(0);
				rx.setFM(minfreq, maxfreq, freq, step);
				rx.setSeekFmLimits(minfreq, maxfreq);
				rx.setFMDeEmphasis(1);				// 1:50us(Japan) 2:75us(USA)
				rxMode=FM;	rxStatus=0;
				rxFreq=rxfreq=freq;
				bfoOn=ssbLoaded=false;
			}else if(data[3]==0x07){				// RXsetSSB(minfreq, maxfreq, freq, step, ul)
				uint16_t minfreq=data[4]|(data[5]<<7)|((data[6]&0x03)<<14);
				uint16_t maxfreq=data[7]|(data[8]<<7)|((data[9]&0x03)<<14);
				uint16_t freq=data[10]|(data[11]<<7)|((data[12]&0x03)<<14);
				uint16_t step=data[13];
				uint8_t ul=(data[12]&0x0C)>>2;		// ul 1:LSB 2:USB
				if(!ssbLoaded)	loadSSB();
				rx.setSSB(minfreq, maxfreq, freq, step, ul);
				rxMode=(ul==1?LSB:USB);	rxStatus=0;
				rxFreq=rxfreq=freq;
				bfoOn=ssbLoaded=false;
			}else if(data[3]==0x08){			// Range for seeking
				uint16_t minfreq=data[4]|(data[5]<<7)|((data[6]&0x03)<<14);
				uint16_t maxfreq=data[7]|(data[8]<<7)|((data[9]&0x03)<<14);
				if(data[6]&0x04)	rx.setSeekFmLimits(minfreq, maxfreq);
				else							rx.setSeekAmLimits(minfreq, maxfreq);
			}else if(data[3]==0x09){			// Seek spacing
				if(data[4]&0x40)	rx.setSeekFmSpacing(data[4]&0x3F);				// 0-63
				else							rx.setSeekAmSpacing(data[4]);							// 0-63
			}else if(data[3]==0x0A){			// RSSI threshold
				if(data[4]&0x40)	rx.setSeekFmRssiThreshold(data[4]&0x3F);	// 0-63
				else							rx.setSeekAmRssiThreshold(data[4]);				// 0-63
			}else if(data[3]==0x0B){			// SNR threshold
				if(data[4]&0x40)	rx.setSeekFmSNRThreshold(data[4]&0x3F);		// 0-63
				else							rx.setSeekAmSNRThreshold(data[4]);				// 0-63
			}else if(data[3]==0x0C){		// RXsetAmSoftMuteMaxAttenuation()
			}else if(data[3]==0x0D){		// RXsetAvcAmMaxGain(idx)
			}else if(data[3]==0x0E){		// RXsetFifoCount(1)
			}else if(data[3]==0x0F){		// RXsetFmBandwidth(idx)
			}
#endif	// -----------------------------------------------------------
		}
	}
}
#if defined(_XIAOM0)	// ---------------------------------------------
void loadSSB(){
	rx.setI2CFastModeCustom(500000);		// 500000 (500kHz)...663ms
	rx.loadPatch(ssb_patch_content, size_content);	// BW 1:2.2kHz
	rx.setI2CStandardMode();						// 100000 (100kHz)
	ssbLoaded=true;
}
void seekFreq(uint16_t freq){	rxfreq=freq;}
#endif	// -----------------------------------------------------------
// --- sysEx の送信 --------------------------------------------------
void sendSysEx(byte* data, int length){
	Serial.print("send: ");	
	for(uint16_t i=0; i<length; i++)	Hexprint(data[i]);
	Serial.println();
	MIDI.sendSysEx(length, data, true);
}
// *** setup *******************************************************************
void setup(){
	Serial.begin(115200);		//	Serial.begin(9600);
	while(!Serial);
	delay(100);
	Wire.begin();
	pinReset();
	lastMillis=millis();
#if defined(_BLEMIDI)
	BLEMIDI.setHandleConnected(BLEconnected);
	BLEMIDI.setHandleDisconnected(BLEdisconnected);
#endif
//MIDI.setHandleNoteOn(handleNoteOn);								// 0x8x
//MIDI.setHandleNoteOff(handleNoteOff);							// 0x9x
	MIDI.setHandleSystemExclusive(OnMidiSysEx);				// 0xF0
	MIDI.setHandleSystemReset(pinReset);							// 0xFF
	MIDI.begin();		// Listen for MIDI messages on channel 1
	Serial.println("Arduino setup done!");
#if defined(_LEONARDO)	// -----------------------------------------------------
	DDRD|=0x20;	PORTD&=0xDF;	// set PD5(TxLED) to 0
	DDRB|=0x01;	PORTB&=0x0E;	// set PB0(RxLED) to 0
#endif // ----------------------------------------------------------------------
}
// *** loop ********************************************************************
void loop(){
	MIDI.read();		// *** PCからのmidiメッセージを受信 ***
	unsigned long currMillis=millis();	// 現在時刻
	if(currMillis-lastMillis>20){				// ***** 20ms毎に ************
		lastMillis+=20;
#if defined(_LEONARDO)	// -----------------------------------------------------
		if(isSMP){					// SMPbegin()後
			if(SMPt>0){				// SMP displayに表示
				SMPt--;
				byte py=SMPt%8;	// .. 0 7 .. 1 0
				PORTD=0;				// x1-x8オフ
				PORTB=(1<<py);	// y1-y8を順にオン .. 0x40 0x20 0x10
				PORTD=SMPy[py];	// x1-x8をセット
			}else{	PORTB=0;	PORTD=0;	}	// y1-y8オフ, x1-x8オフ
		}
#endif									// -----------------------------------------------------
#if defined(_XIAOM0)		// -----------------------------------------------------
		if(isRX){
			if(currMillis-aliveMillis>1200){			// 1.2秒以上PCからaliveが来なければ
				rx.reset();	isRX=false;
			}else if(currMillis-freqMillis>100){	// Freqは100ms毎にチェック
				if(rxfreq!=rxFreq){									// seekでfreqが変化していたら
					byte data[]={0xF0, 0x7D, 0x75, 0x03, rxfreq&0x7F, (rxfreq>>7)&0x7F, (rxfreq>>14)&0x03, 0xF7};
					sendSysEx(data, 8);
					rxFreq=rxfreq;
				}
				freqMillis=currMillis;
			}else if(currMillis-rssiMillis>100){	// RSSIは100ms毎にチェック
				rx.getCurrentReceivedSignalQuality();
				uint8_t rssi=rx.getCurrentRSSI()&0x7F;	// 0-127
				if(rssi/2!=rxRssi/2){								// 精度は0-63
					byte data[]={0xF0, 0x7D, 0x75, 0x01, rssi, 0xF7};
					sendSysEx(data, 6);
					rxRssi=rssi;											// 値を記憶
					rssiMillis=currMillis;
				}
			}else{
				rx.getCurrentReceivedSignalQuality();
				uint8_t status=rxStatus&0x70;
				if(rx.getCurrentPilot())	status|=0x01;
				if(rx.isCurrentTuneFM())	status|=0x02;
				if(rx.isCurrentTuneAM())	status|=0x04;
				if(rx.isCurrentTuneSSB())	status|=0x08;
				if(status!=rxStatus){
					byte data[]={0xF0, 0x7D, 0x75, 0x00, status, 0xF7};
					sendSysEx(data, 6);
					rxStatus=status;
				}
			}
		}
#endif									// -----------------------------------------------------
		for(byte pin=0; pin<NPIN; pin++){		// すべてのピンについて
			if(pinStatus[pin]==INPUT || pinStatus[pin]==INPUT_PULLUP){// デジタル入力なら
				byte val=digitalRead(pin);			// 値を読み
				if(val!=PinVal[pin]){						// 変化があれば
// --- PCにデジタル値を送信 ------------------------------------------
					byte data[]={0xF0, 0x7D, 0x70, pin, val, 0xF7};
					sendSysEx(data, 6);
					PinVal[pin]=val;							// 値を記憶
				}
			}else if(pin!=TEMPpin&&pin!=HUMIpin&&pin!=CO2pin
								&&pinStatus[pin]==INPUT_ANALOG){		// 通常のアナログ入力なら
				int val=analogRead(pin);				// 値を読み
				if(val!=PinVal[pin]){						// 変化があれば *** アナログ値を送信 ***
// --- PCにアナログ値を送信 ------------------------------------------
					byte data[]={0xF0, 0x7D, 0x6F, pin, val&0x7F, (val>>7)&0x7F, 0xF7};
					sendSysEx(data, 7);
					PinVal[pin]=val;							// 値を記憶
				}
			}
		}
	}else if(currMillis-measMillis>1000){	// ***** 1秒毎に *******
		measMillis=currMillis;
		measure();													// 結果はco2ave, tempave, humiave
		uint16_t tbuf=(round(tempave*10.)/10.+45.)*(4095./175.);
																				// 12ビット値(復元はtbuf*175/4095-45)
		if(pinStatus[TEMPpin]==INPUT_ANALOG&&tempave>tempL			// tに変化があるか前回
			&&(tbuf!=PinVal[TEMPpin]||currMillis>=lastTxTemp+Ttx)){	// 送信からTtx経過したら
// --- PCにアナログ値(温度)を送信 ------------------------------------
			byte data[]={0xF0, 0x7D, 0x6F, TEMPpin, tbuf&0x7F, (tbuf>>7)&0x7F, 0xF7};
			sendSysEx(data, 7);
			Serial.print("Send TEMP: ");	Serial.println(tempave);
			PinVal[TEMPpin]=tbuf;							// 12ビット値を記憶
			lastTxTemp=currMillis;						// 送信時刻を記憶
		}
		uint16_t hbuf=humiave*(4095./100.);	// 12ビット値(復元はhbuf*100/4095)
		if(pinStatus[HUMIpin]==INPUT_ANALOG&&humiave>=0					// hに変化があるか前回
			&&(hbuf!=PinVal[HUMIpin]||currMillis>=lastTxHumi+Ttx)){	// 送信からTtx経過したら
// --- PCにアナログ値(湿度)を送信 ------------------------------------
			byte data[]={0xF0, 0x7D, 0x6F, HUMIpin, hbuf&0x7F, (hbuf>>7)&0x7F, 0xF7};
			sendSysEx(data, 7);
			Serial.print("Send HUMI: ");	Serial.println(humiave);
			PinVal[HUMIpin]=hbuf;							// 12ビット値を記憶
			lastTxHumi=currMillis;						// 送信時刻を記憶
		}
		uint16_t cbuf=round(co2ave/10.)*5;	// 1/2にして12ビットで表す(0～4095)
		if(cbuf>=4096)	cbuf=4095;					// (復元はcbuf*2: 0-8190)
		if(pinStatus[CO2pin]==INPUT_ANALOG&&co2ave>=0						// co2に変化があるか前回
			&&(cbuf!=PinVal[CO2pin] || currMillis>=lastTxCO2+Ttx)){	// 送信からTtx経過したら
// --- PCにアナログ値(CO2)を送信 ------------------------------------
			byte data[]={0xF0, 0x7D, 0x6F, CO2pin, cbuf&0x7F, (cbuf>>7)&0x7F, 0xF7};
			sendSysEx(data, 7);
			Serial.print("Send CO2: ");	Serial.println(co2ave);
			PinVal[CO2pin]=cbuf;							// 12ビット値を記憶
			lastTxCO2=currMillis;							// 送信時刻を記憶
		}
#if defined(_LEONARDO)||defined(_PROMICRO)	// ---------------------------------
#else
	}else if(IrReceiver.decode()){
// --- IR受信したら --------------------------------------------------
		decode_type_t protocol=IrReceiver.decodedIRData.protocol;
		uint16_t address =IrReceiver.decodedIRData.address;
		uint16_t command =IrReceiver.decodedIRData.command;
		if(protocol==UNKNOWN){	}					// pi();
		else{
// --- IR受信したデータ(IRDATA)をPCに送信 ----------------------------
			address&=0xFFFF;								// 0x0000-0xFFFF
			command&=0xFFFF;								// 0x0000-0xFFFF
			byte data[]={0xF0, 0x7D, 0x73, protocol&0x7F,	// 0PPPPPPP
									address&0x7F, (address>>7)&0x7F,	// 0aaaaaaa(LSB) 0aaaaaaa
									(address>>14),										// 000000aa(MSB)
									command&0x7F, (command>>7)&0x7F,	// 0ccccccc(LSB) 0ccccccc
									(command>>14),										// 000000cc(MSB)
									 0xF7};
			Serial.print("IRreceived: ");	Hexprint(protocol&0xFF);
			Hexprint(address>>8);	Hexprint(address&0xFF);
			Hexprint(command>>8);	Hexprintln(command&0xFF);
			sendSysEx(data, 11);
//		pi();
		}
		IrReceiver.resume();							// IR受信再開!!
#endif																// ---------------------------------------
	}
}
